import { NextResponse } from 'next/server';

// Fallback to DNS-over-HTTPS (DoH) since Port 53 is blocked/unreliable
async function resolveMxDoH(domain: string) {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=MX`, { cache: 'no-store' });
        if (!res.ok) throw new Error(`DoH Error: ${res.statusText}`);
        const data = await res.json();

        if (data.Status !== 0 || !data.Answer) return []; // No records or error

        return data.Answer.map((ans: any) => {
            // Google DoH format for MX: "10 mail.example.com."
            const parts = ans.data.split(' ');
            return {
                priority: parseInt(parts[0], 10),
                exchange: parts[1]
            };
        });
    } catch (e) {
        console.error("MxDoH Error:", e);
        return [];
    }
}

async function resolveIpDoH(domain: string) {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=A`, { cache: 'no-store' });
        if (!res.ok) return null;
        const data = await res.json();

        if (data.Status === 0 && data.Answer && data.Answer.length > 0) {
            // Return first A record
            return data.Answer.find((a: any) => a.type === 1)?.data || null;
        }
        return null;
    } catch (e) {
        return null;
    }
}

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        const isIp = /^(\d{1,3}\.){3}\d{1,3}$/.test(domain) || domain.includes(':');
        if (isIp) {
            return NextResponse.json({ error: 'MX Lookup requires a domain name, not an IP address.' }, { status: 400 });
        }

        // 1. Resolve MX Records using DoH
        const mxRecords = await resolveMxDoH(domain);

        if (!mxRecords || mxRecords.length === 0) {
            return NextResponse.json({ error: 'No MX records found for this domain.' }, { status: 404 });
        }

        // 2. Resolve IPs and Enrich Data
        const enrichedRecords = await Promise.all(mxRecords.map(async (record: { priority: number, exchange: string }) => {
            let ip: string | null = null;
            let isReachable = true; // Assume reachable if DNS resolved via DoH

            // Resolve IP via DoH
            ip = await resolveIpDoH(record.exchange);
            if (!ip) isReachable = false;

            // Provider Detection
            const exchange = record.exchange.toLowerCase();
            let provider = "Unknown";

            if (exchange.includes("google") || exchange.includes("gmail")) provider = "Google Workspace";
            else if (exchange.includes("outlook") || exchange.includes("microsoft") || exchange.includes("protection.outlook.com")) provider = "Microsoft 365";
            else if (exchange.includes("zoho")) provider = "Zoho Mail";
            else if (exchange.includes("mimecast")) provider = "Mimecast";
            else if (exchange.includes("proton")) provider = "Proton Mail";
            else if (exchange.includes("pphosted")) provider = "Proofpoint";
            else if (exchange.includes("barracuda")) provider = "Barracuda";
            else if (exchange.includes("cloudflare")) provider = "Cloudflare Email";

            return {
                priority: record.priority,
                exchange: record.exchange,
                ip,
                provider,
                isReachable
            };
        }));

        return NextResponse.json({
            domain,
            records: enrichedRecords
        });

    } catch (error: any) {
        console.error("MX Lookup Error:", error);
        return NextResponse.json({ error: `Lookup failed: ${error.message}` }, { status: 500 });
    }
}
