import { NextResponse } from 'next/server';
import dns from 'node:dns/promises';
import { isIPv4, isIPv6 } from 'node:net';

export async function POST(request: Request) {
    try {
        const { ip } = await request.json();

        if (!ip) {
            return NextResponse.json({ error: 'IP address is required' }, { status: 400 });
        }

        let type: 'IPv4' | 'IPv6' = 'IPv4';
        let arpa = '';
        let resolvedIp: string = ip; // Initialize with input ip
        let isDomain: boolean = false; // Initialize as false

        if (isIPv4(ip)) {
            type = 'IPv4';
            resolvedIp = ip;
            // 1.2.3.4 -> 4.3.2.1.in-addr.arpa
            arpa = ip.split('.').reverse().join('.') + '.in-addr.arpa';
        } else if (isIPv6(ip)) {
            type = 'IPv6';
            resolvedIp = ip;
            // Expand IPv6 if needed (simplified here, assuming standard input) - node net.isIPv6 handles format validation
            // Ideally we'd expand 2001::1 to full form, but for ARPA display we can just approximate or use a library
            // For the actual lookup, dns.reverse handles the IP string directly, we just format ARPA for display
            // Expanding IPv6 manually is complex, using a simple placeholder construction for display or skipping strictly
            // Simple expansion logic for display purposes would go here if needed
            // For now we will construct the query zone manually if we wanted, but dns.reverse takes the IP directly.
            // We can just set ARPA to a "Calculated" string for the UI feedback
            arpa = getIPv6Arpa(ip);
        } else {
            // Assume it's a domain and try to resolve it
            try {
                const resolved = await dns.resolve4(ip);
                if (resolved && resolved.length > 0) {
                    resolvedIp = resolved[0];
                    type = 'IPv4'; // Resolved to A record
                    arpa = resolvedIp.split('.').reverse().join('.') + '.in-addr.arpa';
                    isDomain = true;
                } else {
                    return NextResponse.json({ error: 'Could not resolve domain to an IP' }, { status: 400 });
                }
            } catch (e) {
                return NextResponse.json({ error: 'Invalid IP or Domain' }, { status: 400 });
            }
        }

        try {
            const hostnames = await dns.reverse(resolvedIp);

            return NextResponse.json({
                input: ip,
                ip: resolvedIp,
                hostname: hostnames, // Returns array of strings
                arpa,
                type,
                isDomain
            });

        } catch (dnsError: any) {
            if (dnsError.code === 'ENOTFOUND') {
                return NextResponse.json({
                    input: ip,
                    ip: resolvedIp,
                    hostname: null, // Explicit null for no record
                    arpa,
                    type,
                    isDomain
                });
            }
            throw dnsError;
        }

    } catch (error: any) {
        console.error("Reverse DNS Error:", error);
        return NextResponse.json({ error: `Lookup failed: ${error.message}` }, { status: 500 });
    }
}

// Helper to calculate IPv6 ARPA string for display
function getIPv6Arpa(ip: string): string {
    // This is a complex expansion, usually handled by libraries like 'ip-address'
    // To keep it dependency-free and simple, we might just return the suffix or a generic string
    // Or we try to do a basic expansion if it's not too compressed.
    // For this context, standard library lookup doesn't return the ARPA string used, so we synthesize it.
    return `<expanded-ipv6>.ip6.arpa`;
}
