import { NextResponse } from 'next/server';

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        // Clean domain
        let target = domain.replace(/^(https?:\/\/)/, '').split('/')[0];

        // Use DNS-over-HTTPS (DoH) - Google Public DNS
        // Query Root, _dmarc, and default._bimi in parallel to get "All" relevant TXT types
        const endpoints = [
            { name: target, query: target },
            { name: `_dmarc.${target}`, query: `_dmarc.${target}` },
            { name: `default._bimi.${target}`, query: `default._bimi.${target}` }
        ];

        const promises = endpoints.map(ep =>
            fetch(`https://dns.google/resolve?name=${ep.query}&type=TXT`, {
                headers: { 'Accept': 'application/dns-json' }
            }).then(res => res.json())
        );

        const results = await Promise.all(promises);
        const allRecords: string[] = [];

        results.forEach((data) => {
            // Google DNS Status Codes: 0 = NOERROR
            if (data.Status === 0 && data.Answer) {
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                data.Answer.forEach((ans: any) => {
                    // Start/End quote cleanup
                    const clean = ans.data.replace(/^"|"$/g, '').replace(/\\"/g, '"');
                    // Avoid duplicates
                    if (!allRecords.includes(clean)) {
                        allRecords.push(clean);
                    }
                });
            }
        });

        if (allRecords.length === 0) {
            return NextResponse.json({ error: 'No TXT records found' }, { status: 404 });
        }

        return NextResponse.json({ records: allRecords });

    } catch (error: any) {
        console.error("TXT Lookup Error:", error);
        return NextResponse.json({ error: 'Failed to fetch TXT records. Please try again.' }, { status: 500 });
    }
}
