import { NextResponse } from 'next/server';
import net from 'node:net';

// Basic Whois Client (simplified version of api/whois/route.ts)
async function lookupWhois(domain: string, server: string = 'whois.iana.org'): Promise<string> {
    return new Promise((resolve, reject) => {
        const socket = new net.Socket();
        let data = '';

        // Timeout
        const timeout = setTimeout(() => {
            socket.destroy();
            reject(new Error("Timeout connecting to WHOIS server"));
        }, 5000);

        socket.connect(43, server, () => {
            socket.write(`${domain}\r\n`);
        });

        socket.on('data', (chunk) => {
            data += chunk;
        });

        socket.on('end', () => {
            clearTimeout(timeout);
            resolve(data);
        });

        socket.on('error', (err) => {
            clearTimeout(timeout);
            reject(err);
        });
    });
}

// Helper to find referral server
function getReferralServer(data: string): string | null {
    // Strictly look for a hostname (must have a dot) to avoid capturing text labels like "Domain Name:"
    const match = data.match(/(?:whois server|refer|referral server):\s*([a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/i);
    return match ? match[1] : null;
}

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        // 1. Recursive Fetch Logic (Simplistic: IANA -> Registrar)
        let rawData = await lookupWhois(domain);
        const referral = getReferralServer(rawData);

        let registrar = 'Unknown';

        // If referral exists, query it for full data
        if (referral) {
            rawData = await lookupWhois(domain, referral);
            registrar = referral;
        }

        // 2. Date Parsing
        // Common Patterns:
        // Creation Date: 2020-01-01T... | Created on: ... | Creation date: ...
        // Registry Expiry Date: ... | Expiration Date: ... | Expiry date: ...

        const extractDate = (text: string, patterns: RegExp[]): string | null => {
            for (const p of patterns) {
                const match = text.match(p);
                if (match && match[1]) return match[1];
            }
            return null;
        };

        const creationPatterns = [
            /(?:Creation Date|Created On|Registration Date|Created|Domain Name Commencement Date):\s*(.+)/i,
            /(?:created):\s*(.+)/i
        ];
        const expiryPatterns = [
            /(?:Registry Expiry Date|Expiration Date|Expiry Date|Registrar Registration Expiration Date|Paid-Till):\s*(.+)/i,
            /(?:expires):\s*(.+)/i
        ];
        const updatedPatterns = [
            /(?:Updated Date|Last Updated|Domain Last Updated Date):\s*(.+)/i,
            /(?:changed):\s*(.+)/i
        ];

        const createdStr = extractDate(rawData, creationPatterns);
        const expiryStr = extractDate(rawData, expiryPatterns);
        const updatedStr = extractDate(rawData, updatedPatterns);

        if (!createdStr || !expiryStr) {
            // Fallback or error if essential dates missing (typical for availability checks)
            // Check if "No match"
            if (rawData.includes("No match for") || rawData.includes("NOT FOUND")) {
                return NextResponse.json({ error: 'Domain not found or not registered' }, { status: 404 });
            }
            // If registered but parse fail
            return NextResponse.json({
                error: 'Could not parse domain dates. Values may be hidden by privacy.',
                raw: rawData.substring(0, 500)
            }, { status: 422 });
        }

        // 3. Normalize Dates
        const createdDate = new Date(Date.parse(createdStr));
        const expiryDate = new Date(Date.parse(expiryStr));
        const updatedDate = updatedStr ? new Date(Date.parse(updatedStr)) : new Date(); // Fallback to now if missing

        const now = new Date();

        // 4. Calculations
        const msPerYear = 1000 * 60 * 60 * 24 * 365.25;
        const msPerDay = 1000 * 60 * 60 * 24;

        const ageYears = (now.getTime() - createdDate.getTime()) / msPerYear;
        const daysToExpiry = Math.ceil((expiryDate.getTime() - now.getTime()) / msPerDay);

        // 5. Status
        let status: 'Active' | 'Grace Period' | 'Redemption' | 'Expired' = 'Active';
        if (daysToExpiry < 0) status = 'Expired';
        else if (daysToExpiry < 30) status = 'Grace Period'; // Simplification

        // Extract Registrar Name if possible
        const registrarMatch = rawData.match(/Registrar:\s*(.+)/i) || rawData.match(/Sponsoring Registrar:\s*(.+)/i);
        if (registrarMatch) registrar = registrarMatch[1];

        return NextResponse.json({
            domain,
            createdDate: createdDate.toISOString(),
            expiryDate: expiryDate.toISOString(),
            updatedDate: updatedDate.toISOString(),
            ageYears,
            daysToExpiry,
            status,
            registrar
        });

    } catch (error: any) {
        console.error("Domain Expiry Error:", error);
        return NextResponse.json({ error: `Lookup failed: ${error.message}` }, { status: 500 });
    }
}
