import { NextResponse } from 'next/server';

// Helper: Resolve via Google DoH (Bypasses local UDP 53 blocks)
async function resolveDoH(domain: string, type: 'MX' | 'TXT') {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=${type}`, { cache: 'no-store' });
        if (!res.ok) throw new Error(`DoH Status: ${res.status}`);
        const data = await res.json();

        if (data.Status !== 0) return []; // DNS Error or NXDOMAIN
        return data.Answer || [];
    } catch (e) {
        console.error(`DoH ${type} Error:`, e);
        return [];
    }
}

// Helper: Check Blacklist via DoH (Multiple RBLs)
async function checkBlacklistDoH(ip: string): Promise<string | null> {
    try {
        const reversed = ip.split('.').reverse().join('.');

        // 1. Try Spamhaus (Zen)
        // Note: Spamhaus often blocks public resolvers, but we try anyway.
        let answers = await resolveDoH(`${reversed}.zen.spamhaus.org`, 'A' as any);
        if (answers && answers.some((a: any) => a.data.startsWith('127.0.0.'))) {
            return "Spamhaus (Zen)";
        }

        // 2. Try SpamCop (Backup)
        // Often more permissive with public resolvers
        answers = await resolveDoH(`${reversed}.bl.spamcop.net`, 'A' as any);
        if (answers && answers.some((a: any) => a.data.startsWith('127.0.0.'))) {
            return "SpamCop";
        }

        // 3. Try Barracuda (b.barracudacentral.org)
        answers = await resolveDoH(`${reversed}.b.barracudacentral.org`, 'A' as any);
        if (answers && answers.some((a: any) => a.data.startsWith('127.0.0.'))) {
            return "Barracuda";
        }

        return null; // Clean
    } catch {
        return null;
    }
}

// Resolves A record via DoH to get IP from Hostname
async function resolveIpDoH(hostname: string): Promise<string | null> {
    const answers = await resolveDoH(hostname, 'A' as any); // 'A' type is actually 1
    if (answers && answers.length > 0) {
        // Return first A record data (IP)
        return answers.find((a: any) => a.type === 1)?.data || null;
    }
    return null;
}

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();
        if (!domain) return NextResponse.json({ error: 'Domain required' }, { status: 400 });

        const checks: any[] = [];
        const recommendations: string[] = [];
        let score = 0;

        // Parallel DoH Checks
        const [mxAnswers, txtAnswers, dmarcAnswers] = await Promise.all([
            resolveDoH(domain, 'MX'),
            resolveDoH(domain, 'TXT'),
            resolveDoH(`_dmarc.${domain}`, 'TXT')
        ]);

        // 1. MX Check
        let primaryIp = null;
        if (mxAnswers.length > 0) {
            score += 30;
            checks.push({ id: 'mx', name: 'MX Records', status: 'pass', message: `${mxAnswers.length} Mail Exchangers found.` });

            // Parse MX records to find primary
            const parsedMx = mxAnswers
                .filter((a: any) => a.type === 15) // MX type is 15
                .map((a: any) => {
                    const parts = a.data.split(' ');
                    return { priority: parseInt(parts[0]), exchange: parts[1] };
                })
                .sort((a: any, b: any) => a.priority - b.priority);

            // Resolve IP of primary MX
            if (parsedMx.length > 0) {
                primaryIp = await resolveIpDoH(parsedMx[0].exchange);
            }

        } else {
            checks.push({ id: 'mx', name: 'MX Records', status: 'fail', message: 'No valid MX records found.' });
            recommendations.push("Configure MX records to receive email.");
        }

        // 2. SPF Check
        let spfFound = false;
        if (txtAnswers.length > 0) {
            const spfRecord = txtAnswers.find((a: any) => {
                const txt = a.data ? a.data.replace(/^"|"$/g, '') : '';
                return txt.includes('v=spf1');
            });

            if (spfRecord) {
                const txt = spfRecord.data ? spfRecord.data.replace(/^"|"$/g, '') : '';
                score += 30;
                spfFound = true;
                checks.push({ id: 'spf', name: 'SPF Record', status: 'pass', message: 'Valid SPF record detected.', rawValue: txt });
            }
        }

        if (!spfFound) {
            checks.push({ id: 'spf', name: 'SPF Record', status: 'fail', message: 'Missing SPF record.' });
            recommendations.push("Add an SPF record (v=spf1) to authorize senders.");
        }

        // 3. DMARC Check
        let dmarcFound = false;
        if (dmarcAnswers.length > 0) {
            const dmarcRecord = dmarcAnswers.find((a: any) => {
                const txt = a.data ? a.data.replace(/^"|"$/g, '') : '';
                return txt.trim().toUpperCase().startsWith('V=DMARC1');
            });

            if (dmarcRecord) {
                dmarcFound = true;
                const txt = dmarcRecord.data.replace(/^"|"$/g, '');
                const policy = txt.match(/p=(\w+)/)?.[1] || 'none';

                if (policy === 'reject') {
                    score += 30;
                    checks.push({ id: 'dmarc', name: 'DMARC Policy', status: 'pass', message: 'Maximum Protection (p=reject).', rawValue: txt });
                } else if (policy === 'quarantine') {
                    score += 25;
                    checks.push({ id: 'dmarc', name: 'DMARC Policy', status: 'warn', message: 'Moderate Protection (p=quarantine).', rawValue: txt });
                    recommendations.push(`Upgrade DMARC to Reject: "v=DMARC1; p=reject; rua=mailto:dmarc@${domain};"`);
                } else {
                    score += 15;
                    checks.push({ id: 'dmarc', name: 'DMARC Policy', status: 'warn', message: 'Monitoring Only (p=none).', rawValue: txt });
                    recommendations.push(`Set DMARC to Quarantine: "v=DMARC1; p=quarantine; rua=mailto:dmarc@${domain};"`);
                }
            }
        }

        if (!dmarcFound) {
            checks.push({ id: 'dmarc', name: 'DMARC Record', status: 'fail', message: 'DMARC record is missing.' });
            recommendations.push(`Add DMARC Record: "v=DMARC1; p=quarantine; rua=mailto:dmarc@${domain};"`);
        }

        // 4. DKIM (Placeholder)
        checks.push({ id: 'dkim', name: 'DKIM Signature', status: 'warn', message: 'Selector unknown. Verify manually.' });

        // 5. Blacklist (Check Primary MX IP)
        if (primaryIp) {
            const listedOn = await checkBlacklistDoH(primaryIp);

            if (listedOn) {
                // listed
                checks.push({ id: 'blacklist', name: 'Blacklist Status', status: 'fail', message: `Listed on ${listedOn}!` });
                score -= 20; // Penalize for blacklist
                recommendations.push(`Request delisting from ${listedOn} for IP ${primaryIp}.`);
            } else {
                // clean (or skipped/clean via DoH)
                score += 10;
                checks.push({ id: 'blacklist', name: 'Blacklist Status', status: 'pass', message: 'No issues detected on major RBLs.' });
            }

        } else {
            // If checking MX failed, we don't penalize heavily but show a warning
            checks.push({ id: 'blacklist', name: 'Blacklist Status', status: 'warn', message: 'Could not resolve MX IP to check.' });
        }

        return NextResponse.json({
            score: Math.max(0, Math.min(100, score)), // Ensure 0-100
            checks,
            recommendations
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
