import { NextResponse } from 'next/server';

export async function POST(request: Request) {
    try {
        const { url, method = 'GET' } = await request.json();

        if (!url) {
            return NextResponse.json({ error: 'URL is required' }, { status: 400 });
        }

        // 1. Basic SSRF Protection (Block obvious private IPs in URL)
        // Note: Full DNS resolution check is safer, but this catches basic attempts
        // Matches 127.x.x.x, 192.168.x.x, 10.x.x.x, 172.16-31.x.x, localhost
        const privateIpRegex = /^(?:https?:\/\/)?(?:localhost|127\.|192\.168\.|10\.|172\.(?:1[6-9]|2[0-9]|3[0-1])\.|\[::1\]).*/i;
        if (privateIpRegex.test(url)) {
            return NextResponse.json({ error: 'Private/Local networks are restricted.' }, { status: 403 });
        }

        // Add protocol if missing
        let targetUrl = url;
        if (!targetUrl.startsWith('http://') && !targetUrl.startsWith('https://')) {
            targetUrl = 'https://' + targetUrl;
        }

        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 5000); // 5s timeout

        try {
            const response = await fetch(targetUrl, {
                method,
                redirect: 'follow',
                signal: controller.signal,
                headers: {
                    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                    'Accept': '*/*'
                }
            });
            clearTimeout(timeoutId);

            // Categorize Headers
            const headers: Array<{ key: string, value: string, category: string }> = [];
            const rawLines: string[] = [`${response.status} ${response.statusText}`];

            response.headers.forEach((value, key) => {
                const k = key.toLowerCase();
                let category = 'other';

                // Security
                if (['content-security-policy', 'strict-transport-security', 'x-frame-options', 'x-content-type-options', 'referrer-policy', 'permissions-policy'].includes(k)) {
                    category = 'security';
                }
                // Caching
                else if (['cache-control', 'expires', 'vary', 'etag', 'last-modified', 'age'].includes(k)) {
                    category = 'caching';
                }
                // Server
                else if (['server', 'date', 'connection', 'content-type', 'content-encoding', 'content-length'].includes(k)) {
                    category = 'server';
                }

                // Format Key (Capitalize) - Fetch lowercases them, we want pretty
                const formattedKey = key.split('-').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join('-');

                headers.push({ key: formattedKey, value, category });
                rawLines.push(`${formattedKey}: ${value}`);
            });

            return NextResponse.json({
                status: response.status,
                statusText: response.statusText,
                server: response.headers.get('server') || 'Unknown',
                headers,
                rawHeaders: rawLines.join('\n')
            });

        } catch (fetchError: any) {
            clearTimeout(timeoutId);
            if (fetchError.name === 'AbortError') {
                return NextResponse.json({ error: 'Request timed out (5s limit)' }, { status: 408 });
            }
            throw fetchError;
        }

    } catch (error: any) {
        console.error("Header Analysis Error:", error);
        return NextResponse.json({ error: `Failed to fetch URL: ${error.message}` }, { status: 500 });
    }
}
