import { NextResponse } from 'next/server';
import net from 'net';
import dns from 'dns/promises';

export interface PortResult {
    port: number;
    service: string;
    status: 'open' | 'closed' | 'filtered';
    latency: string;
}

// Common service names
const SERVICES: Record<number, string> = {
    21: 'FTP', 22: 'SSH', 23: 'Telnet', 25: 'SMTP', 53: 'DNS',
    80: 'HTTP', 110: 'POP3', 143: 'IMAP', 443: 'HTTPS', 445: 'SMB',
    3306: 'MySQL', 3389: 'RDP', 5432: 'PostgreSQL', 6379: 'Redis',
    8080: 'HTTP-Alt', 27017: 'MongoDB'
};

export async function POST(request: Request) {
    try {
        const { target, ports } = await request.json();

        if (!target || !ports || !Array.isArray(ports)) {
            return NextResponse.json({ error: 'Target and ports array are required' }, { status: 400 });
        }

        if (ports.length > 50) {
            return NextResponse.json({ error: 'Max 50 ports per request' }, { status: 400 });
        }

        // 1. Resolve to IP to check safety
        let ip = target;
        if (!net.isIP(target)) {
            try {
                const lookup = await dns.lookup(target);
                ip = lookup.address;
            } catch {
                return NextResponse.json({ error: 'Could not resolve domain' }, { status: 400 });
            }
        }

        // 2. Safety Check (Block Private IPs)
        if (isPrivateIP(ip)) {
            return NextResponse.json({ error: 'Scanning private/local networks is not allowed' }, { status: 403 });
        }

        // 3. Scan Concurrent
        const results = await Promise.all(ports.map(port => checkPort(ip, port)));

        return NextResponse.json({ target: ip, results });

    } catch (error) {
        console.error("Scan Error:", error);
        return NextResponse.json({ error: 'Scan failed' }, { status: 500 });
    }
}

function isPrivateIP(ip: string): boolean {
    // Basic checks for 127.*, 10.*, 192.168.*, 172.16-31.*
    if (ip.startsWith('127.')) return true; // Loopback
    if (ip.startsWith('10.')) return true; // Private Class A
    if (ip.startsWith('192.168.')) return true; // Private Class C
    if (ip === '::1') return true; // IPv6 Loopback

    const parts = ip.split('.').map(Number);
    if (parts[0] === 172 && parts[1] >= 16 && parts[1] <= 31) return true; // Private Class B

    return false;
}

function checkPort(host: string, port: number): Promise<PortResult> {
    return new Promise((resolve) => {
        const socket = new net.Socket();
        const start = performance.now();
        let status: 'open' | 'closed' = 'closed';

        // 200ms Timeout as requested
        socket.setTimeout(300); // Slight buffer over 200 to imply strictness but allow jitter

        socket.on('connect', () => {
            status = 'open';
            socket.destroy();
        });

        socket.on('timeout', () => {
            status = 'closed'; // Filtered effectively usually behaves like timeout or connection refused
            socket.destroy();
        });

        socket.on('error', () => {
            // connection reset/refused -> closed
            socket.destroy();
        });

        socket.on('close', () => {
            const time = Math.ceil(performance.now() - start);
            // If closed very fast, it's explicitly closed (RST). If timeout, it's filtered/closed.
            // We'll simplify to 'closed' for web UI unless sure.

            resolve({
                port,
                service: SERVICES[port] || 'Unknown',
                status: status,
                latency: status === 'open' ? `${time}ms` : '-'
            });
        });

        socket.connect(port, host);
    });
}
