import { NextResponse } from 'next/server';
import * as cheerio from 'cheerio';

export async function POST(request: Request) {
    try {
        const { url } = await request.json();
        if (!url) return NextResponse.json({ error: 'URL required' }, { status: 400 });

        const targetUrl = url.startsWith('http') ? url : `https://${url}`;
        const parsed = new URL(targetUrl);
        const hostname = parsed.hostname.toLowerCase();

        // Basic SSRF Prevention
        if (hostname === 'localhost' || hostname.startsWith('127.') || hostname.startsWith('192.168.') || hostname === '::1' || hostname.startsWith('10.')) {
            return NextResponse.json({ error: 'Internal/Private targets are not allowed' }, { status: 403 });
        }

        // Fetch HTML
        const res = await fetch(targetUrl, {
            headers: {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36bot'
            },
            signal: AbortSignal.timeout(5000)
        });

        if (!res.ok) {
            throw new Error(`Failed to fetch: ${res.status} ${res.statusText}`);
        }

        const html = await res.text();
        const $ = cheerio.load(html);

        // Helper to get meta content
        const getMeta = (name: string) =>
            $(`meta[name="${name}"]`).attr('content') ||
            $(`meta[property="${name}"]`).attr('content') ||
            $(`meta[name="twitter:${name}"]`).attr('content') ||
            $(`meta[property="twitter:${name}"]`).attr('content') || '';

        // Helper to resolve URL
        const resolveUrl = (path?: string) => {
            if (!path) return '';
            try {
                return new URL(path, targetUrl).toString();
            } catch {
                return path;
            }
        };

        const title = $('title').text() || getMeta('og:title') || getMeta('twitter:title');
        const description = getMeta('description') || getMeta('og:description') || getMeta('twitter:description');
        const image = resolveUrl(getMeta('og:image') || getMeta('twitter:image'));
        const favicon = resolveUrl($('link[rel="icon"]').attr('href') || $('link[rel="shortcut icon"]').attr('href') || '/favicon.ico');
        const canonical = resolveUrl($('link[rel="canonical"]').attr('href')) || targetUrl;
        const siteName = getMeta('og:site_name');

        // --- SEO ANALYSIS ---
        const h1Count = $('h1').length;
        const suggestions: { severity: 'critical' | 'warning' | 'info', message: string, fix: string }[] = [];
        let score = 100;

        // 1. Title Analysis
        if (!title) {
            score -= 20;
            suggestions.push({ severity: 'critical', message: 'Missing Title Tag', fix: 'Add a <title> tag to the <head> section.' });
        } else if (title.length < 10) {
            score -= 5;
            suggestions.push({ severity: 'warning', message: 'Title too short', fix: 'Expand title to at least 10 characters for better context.' });
        } else if (title.length > 60) {
            score -= 5;
            suggestions.push({ severity: 'warning', message: 'Title too long', fix: 'Shorten title to under 60 characters to avoid truncation in SERPs.' });
        }

        // 2. Description Analysis
        if (!description) {
            score -= 20;
            suggestions.push({ severity: 'critical', message: 'Missing Meta Description', fix: 'Add a <meta name="description"> tag.' });
        } else if (description.length < 50) {
            score -= 5;
            suggestions.push({ severity: 'warning', message: 'Description too short', fix: 'Expand description to at least 50 characters.' });
        } else if (description.length > 160) {
            score -= 5;
            suggestions.push({ severity: 'warning', message: 'Description too long', fix: 'Keep description under 160 characters.' });
        }

        // 3. Social Tags
        if (!getMeta('og:image')) {
            score -= 10;
            suggestions.push({ severity: 'warning', message: 'Missing Open Graph Image', fix: 'Add <meta property="og:image"> for social sharing previews.' });
        }
        if (!getMeta('twitter:card')) {
            score -= 5;
            suggestions.push({ severity: 'info', message: 'Missing Twitter Card', fix: 'Add <meta name="twitter:card" content="summary_large_image">.' });
        }

        // 4. Structure
        if (h1Count === 0) {
            score -= 10;
            suggestions.push({ severity: 'critical', message: 'Missing H1 Heading', fix: 'Add exactly one <h1> tag to structure your page content.' });
        } else if (h1Count > 1) {
            score -= 5;
            suggestions.push({ severity: 'warning', message: 'Multiple H1 Tags', fix: 'Use only one <h1> tag per page for main title.' });
        }

        if (!favicon) {
            score -= 5;
            suggestions.push({ severity: 'info', message: 'Missing Favicon', fix: 'Add a <link rel="icon"> tag.' });
        }

        const data = {
            title: title.trim(),
            description: description.trim(),
            image,
            url: canonical,
            ogType: getMeta('og:type') || 'website',
            twitterCard: getMeta('twitter:card') || 'summary_large_image',
            favicon,
            canonical,
            siteName,
            analysis: {
                score: Math.max(0, score),
                h1Count,
                suggestions
            }
        };

        return NextResponse.json(data);

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
