import { NextResponse } from 'next/server';
import tls from 'tls';
import net from 'net';
import { URL } from 'url';

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        // Clean hostname (remove https://, paths, etc)
        let hostname = domain;
        try {
            if (!hostname.startsWith('http')) {
                hostname = 'https://' + hostname;
            }
            const url = new URL(hostname);
            hostname = url.hostname;
        } catch {
            return NextResponse.json({ error: 'Invalid domain format' }, { status: 400 });
        }

        const details = await getSSLDetails(hostname);
        return NextResponse.json(details);

    } catch (error: any) {
        console.error("SSL Check Error:", error);
        return NextResponse.json({ error: error.message || 'Failed to check SSL' }, { status: 500 });
    }
}

function getSSLDetails(hostname: string): Promise<any> {
    return new Promise((resolve, reject) => {
        const options = {
            host: hostname,
            port: 443,
            servername: hostname, // SNI support
            rejectUnauthorized: false, // We want to inspect bad certs too, not just crash
        };

        const socket = tls.connect(options, () => {
            const cert = socket.getPeerCertificate(true); // true to get full chain if possible

            if (!cert || Object.keys(cert).length === 0) {
                socket.end();
                return reject(new Error('No certificate found'));
            }

            const validTo = new Date(cert.valid_to);
            const validFrom = new Date(cert.valid_from);
            const now = new Date();
            const daysRemaining = Math.max(0, Math.ceil((validTo.getTime() - now.getTime()) / (1000 * 60 * 60 * 24)));

            // Extract Chain
            const chain: string[] = [];
            // Basic recursive chain extraction if 'issuerCertificate' is present
            // However, node's getPeerCertificate(true) returns the cert with a .issuerCertificate property recursively

            let current = cert;
            while (current) {
                // Determine name
                const name = current.subject && current.subject.CN ? current.subject.CN : 'Unknown';
                chain.push(name);

                // Break if self-signed root (issuer == subject)
                if (current.issuerCertificate && current.issuerCertificate !== current) {
                    // Safety break for circular refs although tls module usually handles this
                    if (chain.includes(current.issuerCertificate.subject.CN) && chain.length > 1) break;
                    current = current.issuerCertificate;
                } else {
                    break;
                }
            }
            // The logic above goes Child -> Parent. For display we usually want Root -> Intermediate -> Leaf or Leaf -> Root.
            // "Chain of trust" usually implies Root down to Leaf. 
            // My loop goes Leaf (connect) -> Issuer (Intermediate) -> Issuer (Root).
            // So we should reverse it.
            chain.reverse();

            const info = {
                subject: cert.subject.CN || hostname,
                issuer: cert.issuer.CN || cert.issuer.O || 'Unknown',
                validFrom: cert.valid_from,
                validTo: cert.valid_to,
                daysRemaining,
                serialNumber: cert.serialNumber,
                fingerprint: cert.fingerprint,
                cipher: socket.getCipher().name,
                protocol: socket.getProtocol(),
                chain,
                authorized: socket.authorized,
                error: socket.authorizationError
            };

            socket.end();
            resolve(info);
        });

        socket.setTimeout(5000);
        socket.on('timeout', () => {
            socket.destroy();
            reject(new Error('Connection timed out'));
        });

        socket.on('error', (err) => {
            reject(err);
        });
    });
}
