import { NextResponse } from 'next/server';
import { spawn } from 'node:child_process';
import os from 'node:os';

export const runtime = 'nodejs';

export async function POST(request: Request) {
    try {
        const { host, maxHops = 30 } = await request.json();

        if (!host) {
            return NextResponse.json({ error: 'Host is required' }, { status: 400 });
        }

        const isValidHost = /^[a-zA-Z0-9.-]+$/.test(host);
        if (!isValidHost) {
            return NextResponse.json({ error: 'Invalid hostname format' }, { status: 400 });
        }

        const safeHops = Math.min(Math.max(1, Number(maxHops)), 30);

        const isWindows = os.platform() === 'win32';
        const command = isWindows ? 'tracert' : 'traceroute';

        // Windows: -h <max_hops>, -w <timeout_ms>
        // Linux: -m <max_hops>, -w <timeout_sec>
        const args = isWindows
            ? ['-h', safeHops.toString(), '-w', '1000', host]
            : ['-m', safeHops.toString(), '-w', '1', host];

        const encoder = new TextEncoder();
        const stream = new ReadableStream({
            start(controller) {
                const child = spawn(command, args);

                child.stdout.on('data', (chunk) => {
                    const text = chunk.toString();
                    controller.enqueue(encoder.encode(text));
                });

                child.stderr.on('data', (chunk) => {
                    const text = chunk.toString();
                    controller.enqueue(encoder.encode(text));
                });

                child.on('error', (err) => {
                    controller.enqueue(encoder.encode(`\nError: ${err.message}\n`));
                    controller.close();
                });

                child.on('close', (code) => {
                    controller.close();
                });
            }
        });

        return new Response(stream, {
            headers: {
                'Content-Type': 'text/plain; charset=utf-8',
                'Transfer-Encoding': 'chunked',
                'X-Content-Type-Options': 'nosniff',
            },
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
