import { NextResponse } from 'next/server';

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        const apiKey = process.env.WHOISXML_API_KEY;
        if (!apiKey) {
            console.error("Missing WHOISXML_API_KEY");
            return NextResponse.json({ error: 'Server configuration error' }, { status: 500 });
        }

        const params = new URLSearchParams({
            apiKey: apiKey,
            domainName: domain,
            outputFormat: 'json'
        });

        const response = await fetch(`https://www.whoisxmlapi.com/whoisserver/WhoisService?${params.toString()}`);

        if (!response.ok) {
            throw new Error(`API responded with status: ${response.status}`);
        }

        const data = await response.json();

        if (data.ErrorMessage) {
            return NextResponse.json({ error: data.ErrorMessage.msg || "WHOIS lookup failed" }, { status: 400 });
        }

        const record = data.WhoisRecord;
        if (!record) {
            return NextResponse.json({ error: "No WHOIS record found" }, { status: 404 });
        }

        // Helper to map contact info
        const mapContact = (contact: any) => ({
            name: contact?.name || contact?.organization || "",
            organization: contact?.organization || "",
            street: contact?.street1 || contact?.street || "",
            city: contact?.city || "",
            state: contact?.state || contact?.stateProvince || "",
            postalCode: contact?.postalCode || "",
            country: contact?.country || "",
            phone: contact?.telephone || contact?.tenantTelephone || "",
            email: contact?.email || ""
        });

        // Helper to map status (can be array or string or missing)
        let statusList: string[] = [];
        if (record.status) {
            if (Array.isArray(record.status)) {
                statusList = record.status;
            } else {
                statusList = [record.status];
            }
        }

        // Fallback checks for critical data
        const getLastUpdated = () =>
            record.updatedDate ||
            record.registryData?.updatedDate ||
            record.audit?.updatedDate || "";

        const getCreationDate = () =>
            record.createdDate ||
            record.createdDateNormalized ||
            record.registryData?.createdDate ||
            record.registryData?.createdDateNormalized ||
            record.audit?.createdDate || "";

        const getAllExpirationDates = () =>
            record.expiresDate ||
            record.expiresDateNormalized ||
            record.registryData?.expiresDate ||
            record.registryData?.expiresDateNormalized || "";

        const getRegistrarName = () =>
            record.registrarName ||
            record.registryData?.registrarName ||
            record.registrar?.name || "";

        const getRegistrarIANA = () =>
            record.registrarIANAID ||
            record.registryData?.registrarIANAID ||
            record.registrar?.ianaId || "";

        const getRegistrarURL = () =>
            record.registrarURL ||
            record.registryData?.registrarURL ||
            record.registrar?.url || "";

        const mappedData = {
            domainName: record.domainName || domain,
            registrar: {
                name: getRegistrarName(),
                ianaId: getRegistrarIANA(),
                url: getRegistrarURL(),
                abuseContactEmail: record.contactEmail || record.registrar?.email || "",
                abuseContactPhone: record.registrar?.phone || ""
            },
            whoisServer: record.registryData?.whoisServer || record.whoisServer || "",
            updatedDate: getLastUpdated(),
            creationDate: getCreationDate(),
            expirationDate: getAllExpirationDates(),
            nameservers: record.nameServers?.hostNames || record.registryData?.nameServers?.hostNames || [],
            status: statusList.length > 0 ? statusList : (record.registryData?.status ? [record.registryData.status].flat() : []),
            dnssec: record.dnssec || record.registryData?.dnssec || "",
            registrant: mapContact(record.registrant || record.registryData?.registrant),
            administrative: mapContact(record.administrativeContact || record.registryData?.administrativeContact),
            technical: mapContact(record.technicalContact || record.registryData?.technicalContact),
            billing: mapContact(record.billingContact || record.registryData?.billingContact),
            rawData: record.rawText || record.registryData?.rawText || ""
        };

        return NextResponse.json(mappedData);

    } catch (error) {
        console.error("WHOIS API Error:", error);
        return NextResponse.json({ error: 'Failed to fetch WHOIS data. Please try again.' }, { status: 500 });
    }
}

