"use client";

import React, { useEffect, useState } from "react";
import { useSearchParams } from "next/navigation";
import { motion } from "framer-motion";
import { Globe, Shield, Activity, Server, FileText, AlertTriangle, CheckCircle, XCircle, Loader2, ArrowRight } from "lucide-react";
import { WhoisResult } from "@/components/whois/WhoisResult";
import { DnsResult } from "@/app/actions/dns";
import { checkDnsPropagation } from "@/app/actions/dns";

interface BlacklistResult {
    host: string;
    name: string;
    status: 'clean' | 'listed' | 'error';
    responseTime: number;
}

interface SSLResult {
    subject: string;
    issuer: string;
    validFrom: string;
    validTo: string;
    daysRemaining: number;
    authorized: boolean;
    error?: string;
}

export default function AnalyzerClient() {
    const searchParams = useSearchParams();
    const query = searchParams.get('q');

    const [isLoading, setIsLoading] = useState(true);
    const [progress, setProgress] = useState(0);

    // Data States
    const [whoisData, setWhoisData] = useState<any>(null);
    const [dnsData, setDnsData] = useState<DnsResult[]>([]);
    const [blacklistData, setBlacklistData] = useState<BlacklistResult[]>([]);
    const [sslData, setSslData] = useState<SSLResult | null>(null);
    const [ipAddress, setIpAddress] = useState<string | null>(null);

    // Errors
    const [errors, setErrors] = useState<string[]>([]);

    useEffect(() => {
        if (query) {
            runAnalysis(query);
        } else {
            setIsLoading(false);
        }
    }, [query]);

    const runAnalysis = async (domain: string) => {
        setIsLoading(true);
        setProgress(10);
        setErrors([]);

        // 1. Start parallel fetches
        // We'll increment progress as things finish

        try {
            // A. DNS Propagation (Fastest way to get IP too)
            const dnsPromise = checkDnsPropagation(domain, "A")
                .then(res => {
                    setDnsData(res);
                    setProgress(p => p + 20);
                    // Extract IP from Local or Google for Blacklist check
                    const validIp = res.find(r => r.status === 'ok' && r.value.match(/^\d+\.\d+\.\d+\.\d+$/))?.value.split(',')[0].trim();
                    if (validIp) setIpAddress(validIp);
                    return validIp;
                })
                .catch(e => {
                    console.error("DNS Error", e);
                    setErrors(prev => [...prev, "DNS Lookup Failed"]);
                    return null;
                });

            // B. WHOIS
            const whoisPromise = fetch('/api/whois', {
                method: 'POST',
                body: JSON.stringify({ domain })
            }).then(ifOk).then(res => {
                setWhoisData(res);
                setProgress(p => p + 20);
            }).catch(e => {
                console.error("WHOIS Error", e);
                setErrors(prev => [...prev, "WHOIS Lookup Failed"]);
            });

            // C. SSL
            const sslPromise = fetch('/api/ssl', {
                method: 'POST',
                body: JSON.stringify({ domain })
            }).then(ifOk).then(res => {
                setSslData(res);
                setProgress(p => p + 20);
            }).catch(e => {
                console.error("SSL Error", e);
                // Don't show error for SSL unless critical, as non-HTTPS domains exist
            });

            // Wait for IP to be ready for Blacklist Check
            const ip = await dnsPromise;

            if (ip) {
                // D. Blacklist (Requires IP)
                fetch('/api/blacklist/advanced', {
                    method: 'POST',
                    body: JSON.stringify({ ip })
                }).then(ifOk).then(res => {
                    setBlacklistData(res.results || []);
                    setProgress(p => p + 30);
                }).catch(e => {
                    console.error("Blacklist Error", e);
                });
            } else {
                setProgress(p => p + 30); // Skip blacklist progress
            }

            await Promise.allSettled([whoisPromise, sslPromise]);

        } catch (err) {
            console.error(err);
        } finally {
            setIsLoading(false);
            setProgress(100);
        }
    };

    const ifOk = async (res: Response) => {
        if (!res.ok) throw new Error(res.statusText);
        return res.json();
    }

    // Render Logic
    if (!query) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-slate-50">
                <div className="text-center">
                    <h1 className="text-2xl font-bold text-slate-900">No Domain Specified</h1>
                    <p className="text-slate-500 mb-4">Please verify a domain from the homepage.</p>
                    <a href="/" className="text-blue-600 font-bold hover:underline">Go Home</a>
                </div>
            </div>
        )
    }

    return (
        <div className="min-h-screen bg-[#F8FAFC] pb-20">
            {/* Header */}
            <header className="bg-white border-b border-slate-200 sticky top-0 z-30 shadow-sm">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 h-16 flex items-center justify-between">
                    <div className="flex items-center gap-3">
                        <div className="bg-blue-600 p-2 rounded-lg">
                            <Activity className="w-5 h-5 text-white" />
                        </div>
                        <h1 className="text-xl font-bold text-slate-900">
                            Analysis Report: <span className="text-blue-600">{query}</span>
                        </h1>
                    </div>
                    {isLoading ? (
                        <div className="flex items-center gap-3">
                            <div className="w-32 h-2 bg-slate-100 rounded-full overflow-hidden">
                                <motion.div
                                    className="h-full bg-blue-600"
                                    initial={{ width: 0 }}
                                    animate={{ width: `${progress}%` }}
                                />
                            </div>
                            <span className="text-sm font-bold text-slate-500">{progress}%</span>
                        </div>
                    ) : (
                        <a href="/" className="text-sm font-bold text-slate-500 hover:text-slate-900">New Scan</a>
                    )}
                </div>
            </header>

            <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8 space-y-8">

                {/* 1. Overview Cards */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                    <SummaryCard
                        title="Domain Status"
                        status={whoisData ? "Active" : "Unknown"}
                        icon={Globe}
                        color={whoisData ? "green" : "slate"}
                        detail={whoisData?.registrar?.name || "Registrar Unknown"}
                    />
                    <SummaryCard
                        title="SSL Security"
                        status={sslData?.authorized ? "Secure" : "Not Secure"}
                        icon={Shield}
                        color={sslData?.authorized ? "green" : "red"}
                        detail={sslData ? `${sslData.daysRemaining} Days Left` : "No Certificate"}
                    />
                    <SummaryCard
                        title="Blacklist Check"
                        status={blacklistData.length > 0 ? (blacklistData.some(b => b.status === 'listed') ? "Listed" : "Clean") : "Pending"}
                        icon={AlertTriangle}
                        color={blacklistData.some(b => b.status === 'listed') ? "red" : "green"}
                        detail={blacklistData.length > 0 ? `${blacklistData.filter(b => b.status === 'clean').length}/${blacklistData.length} Clean` : "Checking..."}
                    />
                    <SummaryCard
                        title="IP Address"
                        status={ipAddress || "Resolving..."}
                        icon={Server}
                        color="blue"
                        detail={dnsData.length > 0 ? `${dnsData.filter(d => d.status === 'ok').length} Providers OK` : "Checking..."}
                    />
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Left Column (Main Specs) */}
                    <div className="lg:col-span-2 space-y-8">

                        {/* WHOIS Section */}
                        <Section title="WHOIS Information" icon={FileText}>
                            {whoisData ? (
                                <div className="prose max-w-none">
                                    <WhoisResult data={whoisData} />
                                </div>
                            ) : isLoading ? (
                                <SkeletonLoader />
                            ) : (
                                <ErrorState msg="Could not retrieve WHOIS data." />
                            )}
                        </Section>

                        {/* DNS Section */}
                        <Section title="DNS Propagation" icon={Globe}>
                            {dnsData.length > 0 ? (
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                    {dnsData.map((res, i) => (
                                        <div key={i} className="flex items-center justify-between p-3 bg-white border border-slate-200 rounded-lg">
                                            <div className="flex items-center gap-3">
                                                <span className="text-xl">{res.flag}</span>
                                                <div>
                                                    <div className="font-bold text-sm text-slate-900">{res.provider}</div>
                                                    <div className="text-xs text-slate-500">{res.location}</div>
                                                </div>
                                            </div>
                                            {res.status === 'ok' ? (
                                                <CheckCircle className="w-5 h-5 text-green-500" />
                                            ) : (
                                                <XCircle className="w-5 h-5 text-red-500" />
                                            )}
                                        </div>
                                    ))}
                                </div>
                            ) : isLoading ? <SkeletonLoader /> : <ErrorState msg="No DNS Records Found" />}
                        </Section>
                    </div>

                    {/* Right Column (Security & Blacklist) */}
                    <div className="space-y-8">
                        {/* SSL Section */}
                        <Section title="SSL Certificate" icon={Shield}>
                            {sslData ? (
                                <div className="space-y-4">
                                    <div className={`p-4 rounded-xl border ${sslData.authorized ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'}`}>
                                        <div className="font-bold text-lg mb-1 flex items-center gap-2">
                                            {sslData.authorized ? <CheckCircle className="text-green-600 w-5 h-5" /> : <XCircle className="text-red-600 w-5 h-5" />}
                                            {sslData.authorized ? "Valid Certificate" : "Invalid Certificate"}
                                        </div>
                                        <div className="text-sm opacity-80">
                                            Issued by {sslData.issuer}
                                        </div>
                                    </div>
                                    <div className="space-y-3 text-sm">
                                        <div className="flex justify-between py-2 border-b border-slate-100">
                                            <span className="text-slate-500">Common Name</span>
                                            <span className="font-medium">{sslData.subject}</span>
                                        </div>
                                        <div className="flex justify-between py-2 border-b border-slate-100">
                                            <span className="text-slate-500">Expires In</span>
                                            <span className={`font-medium ${sslData.daysRemaining < 30 ? 'text-red-600' : 'text-green-600'}`}>{sslData.daysRemaining} Days</span>
                                        </div>
                                        <div className="flex justify-between py-2 border-b border-slate-100">
                                            <span className="text-slate-500">Valid From</span>
                                            <span className="font-medium">{new Date(sslData.validFrom).toLocaleDateString()}</span>
                                        </div>
                                        <div className="flex justify-between py-2 border-b border-slate-100">
                                            <span className="text-slate-500">Valid To</span>
                                            <span className="font-medium">{new Date(sslData.validTo).toLocaleDateString()}</span>
                                        </div>
                                    </div>
                                </div>
                            ) : isLoading ? <SkeletonLoader height="h-40" /> : <div className="text-slate-400 italic text-sm">No SSL data available.</div>}
                        </Section>

                        {/* Blacklist Section */}
                        <Section title="Blacklist Check" icon={AlertTriangle}>
                            {blacklistData.length > 0 ? (
                                <div className="space-y-2 max-h-[500px] overflow-y-auto pr-2 scrollbar-thin scrollbar-thumb-slate-200">
                                    {blacklistData.map((bl, i) => (
                                        <div key={i} className="flex items-center justify-between text-sm p-2 rounded hover:bg-slate-50">
                                            <span className="text-slate-700 font-medium truncate w-40" title={bl.name}>{bl.name}</span>
                                            {bl.status === 'listed' ? (
                                                <span className="bg-red-100 text-red-700 text-xs px-2 py-0.5 rounded font-bold">LISTED</span>
                                            ) : bl.status === 'error' ? (
                                                <span className="text-slate-400 text-xs">Error</span>
                                            ) : (
                                                <span className="text-green-600 text-xs flex items-center gap-1"><CheckCircle className="w-3 h-3" /> Clean</span>
                                            )}
                                        </div>
                                    ))}
                                </div>
                            ) : isLoading ? <SkeletonLoader count={5} height="h-8" /> : <div className="text-slate-400 italic text-sm">Waiting for IP resolution...</div>}
                        </Section>
                    </div>
                </div>
            </main>
        </div>
    );
}

const SummaryCard = ({ title, status, icon: Icon, color, detail }: any) => {
    const colorClasses = {
        green: "bg-green-50 text-green-700 border-green-200",
        red: "bg-red-50 text-red-700 border-red-200",
        blue: "bg-blue-50 text-blue-700 border-blue-200",
        slate: "bg-slate-50 text-slate-700 border-slate-200"
    };

    // Default to slate if color key is invalid
    const activeClass = colorClasses[color as keyof typeof colorClasses] || colorClasses.slate;

    return (
        <div className="bg-white border border-slate-200 p-5 rounded-xl shadow-sm">
            <div className="flex items-start justify-between mb-4">
                <div className={`p-2 rounded-lg ${activeClass} bg-opacity-50`}>
                    <Icon className="w-5 h-5" />
                </div>
                {color === 'green' && <CheckCircle className="w-4 h-4 text-green-500" />}
                {color === 'red' && <AlertTriangle className="w-4 h-4 text-red-500" />}
            </div>
            <h3 className="text-slate-500 text-sm font-medium uppercase tracking-wide">{title}</h3>
            <div className="text-2xl font-bold text-slate-900 mt-1 truncate" title={status}>{status}</div>
            <div className="text-xs text-slate-400 mt-2 font-medium">{detail}</div>
        </div>
    );
}

const Section = ({ title, icon: Icon, children }: any) => (
    <section className="bg-white border border-slate-200 rounded-2xl shadow-sm overflow-hidden">
        <div className="border-b border-slate-100 px-6 py-4 flex items-center gap-3 bg-slate-50/50">
            <Icon className="w-5 h-5 text-blue-600" />
            <h2 className="font-bold text-slate-800 text-lg">{title}</h2>
        </div>
        <div className="p-6">
            {children}
        </div>
    </section>
);

const SkeletonLoader = ({ height = "h-64", count = 1 }) => (
    <div className="space-y-3">
        {Array.from({ length: count }).map((_, i) => (
            <div key={i} className={`w-full ${height} bg-slate-100 rounded-xl animate-pulse`} />
        ))}
    </div>
);

const ErrorState = ({ msg }: { msg: string }) => (
    <div className="text-center py-10 bg-red-50 rounded-xl border border-red-100 text-red-600">
        <AlertTriangle className="w-8 h-8 mx-auto mb-2 opacity-50" />
        <p className="font-medium">{msg}</p>
    </div>
);
