"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Shield, ShieldAlert, ShieldCheck, Loader2, Search, Activity, Server, AlertTriangle } from "lucide-react";

import { BlacklistDashboard, BlacklistResult } from "@/components/blacklist-checker/BlacklistDashboard";

export default function BlacklistClient() {
    const [ip, setIp] = useState("");
    const [loading, setLoading] = useState(false);
    const [results, setResults] = useState<BlacklistResult[]>([]);
    const [error, setError] = useState("");
    const [searched, setSearched] = useState(false);

    const handleSearch = async () => {
        if (!ip) return;
        setLoading(true);
        setError("");
        setResults([]);
        setSearched(true);

        try {
            const res = await fetch('/api/blacklist/advanced', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ ip })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to scan blacklists");
            }
            setResults(json.results);

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    // Derived State
    const totalLists = results.length;
    const listed = results.filter(r => r.status === 'listed');
    const listedCount = listed.length;
    const isClean = searched && !loading && listedCount === 0;
    const hasThreats = searched && !loading && listedCount > 0;

    // Reputation Score: 100 - (Listed * 10) for heavier penalty
    const reputationScore = totalLists > 0 ? Math.max(0, 100 - (listedCount * 10)) : 0;
    const avgLatency = totalLists > 0 ? Math.round(results.reduce((acc, r) => acc + r.responseTime, 0) / totalLists) : 0;

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-indigo-100 selection:text-indigo-900">

            {/* Header with Dynamic Gradient */}
            <div className="relative pt-24 pb-32 overflow-hidden bg-white transition-colors duration-1000">
                {/* Background Effects */}
                <div className="absolute inset-0 z-0">
                    <div className="absolute top-0 inset-x-0 h-px bg-gradient-to-r from-transparent via-slate-200 to-transparent" />
                    {/* Dynamic Aurora based on State */}
                    {hasThreats ? (
                        <>
                            <div className="absolute -top-[30%] -left-[10%] w-[70%] h-[70%] rounded-full bg-red-100/40 blur-[100px] animate-pulse" />
                            <div className="absolute top-[10%] -right-[10%] w-[50%] h-[60%] rounded-full bg-orange-100/40 blur-[100px]" />
                        </>
                    ) : isClean ? (
                        <>
                            <div className="absolute -top-[30%] -left-[10%] w-[70%] h-[70%] rounded-full bg-emerald-100/30 blur-[100px]" />
                            <div className="absolute top-[10%] -right-[10%] w-[50%] h-[60%] rounded-full bg-teal-100/30 blur-[100px]" />
                        </>
                    ) : (
                        <>
                            <div className="absolute -top-[30%] -left-[10%] w-[70%] h-[70%] rounded-full bg-indigo-100/30 blur-[100px]" />
                            <div className="absolute top-[10%] -right-[10%] w-[50%] h-[60%] rounded-full bg-blue-100/30 blur-[100px]" />
                        </>
                    )}
                    <div className="absolute inset-0 opacity-[0.03]" style={{ backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23000000' fill-opacity='1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")` }} />
                </div>

                <div className="max-w-4xl mx-auto px-4 text-center relative z-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5 }}>

                        {/* Status Icon */}
                        <div className={`inline-flex items-center justify-center p-3 mb-6 bg-white border shadow-xl rounded-2xl transition-all duration-500
                             ${hasThreats ? "shadow-red-500/20 text-red-600 border-red-100" : isClean ? "shadow-emerald-500/20 text-emerald-600 border-emerald-100" : "shadow-indigo-500/10 text-indigo-600 border-indigo-50"}
                        `}>
                            {hasThreats ? <ShieldAlert className="w-8 h-8 animate-pulse" /> : isClean ? <ShieldCheck className="w-8 h-8" /> : <Shield className="w-8 h-8" />}
                        </div>

                        <h1 className="text-4xl md:text-6xl font-extrabold text-slate-900 mb-6 tracking-tight font-jakarta">
                            Blacklist <span className={`text-transparent bg-clip-text bg-gradient-to-r transition-all duration-500 ${hasThreats ? "from-red-600 to-orange-600" : isClean ? "from-emerald-600 to-teal-600" : "from-indigo-600 to-blue-600"}`}>Audit & Monitor</span>
                        </h1>
                        <p className="text-lg text-slate-500 max-w-2xl mx-auto mb-10 leading-relaxed">
                            Real-time reputation analysis across 50+ enterprise-grade RBLs.
                            Detect blocked emails, compromised hosts, and security threats instantly.
                        </p>

                        {/* Search Interface */}
                        <div className="flex flex-col sm:flex-row items-stretch gap-3 max-w-xl mx-auto bg-white/70 backdrop-blur-xl p-2 rounded-2xl shadow-2xl shadow-slate-900/5 border border-white/50 ring-1 ring-slate-200/50">
                            <div className="flex-1 relative group">
                                <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-indigo-500 transition-colors" />
                                <input
                                    type="text"
                                    placeholder="Enter IP address (e.g. 1.2.3.4)"
                                    value={ip}
                                    onChange={(e) => setIp(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="w-full h-12 pl-12 pr-4 bg-transparent text-lg font-medium text-slate-900 placeholder:text-slate-400 outline-none rounded-xl hover:bg-slate-50/50 focus:bg-slate-50 transition-colors"
                                    suppressHydrationWarning
                                />
                            </div>

                            <button
                                onClick={handleSearch}
                                disabled={loading || !ip}
                                className={`h-12 px-8 font-bold rounded-xl transition-all shadow-lg active:scale-95 disabled:opacity-70 disabled:active:scale-100 flex items-center gap-2 text-white
                                    ${hasThreats ? "bg-red-600 hover:bg-red-700 shadow-red-600/20" : "bg-slate-900 hover:bg-slate-800 shadow-slate-900/20"}
                                `}
                            >
                                {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : "Scan Target"}
                            </button>
                        </div>
                    </motion.div>
                </div>
            </div>

            {/* Content Area */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 -mt-20 relative z-20 pb-20 w-full">

                {error && (
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} className="max-w-xl mx-auto bg-red-50 border border-red-100 text-red-600 p-4 rounded-xl text-center shadow-lg mb-8 font-medium">
                        {error}
                    </motion.div>
                )}

                <AnimatePresence mode="wait">
                    {searched && !loading && (
                        <motion.div
                            key="results"
                            initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ duration: 0.4 }}
                            className="space-y-6"
                        >

                            {/* CRITICAL ALERT HERO - Shows ONLY if threats found */}
                            {hasThreats && (
                                <div className="bg-white rounded-3xl border border-red-100 shadow-xl shadow-red-900/5 overflow-hidden">
                                    <div className="bg-red-50/50 p-6 sm:p-8 border-b border-red-100 flex flex-col sm:flex-row items-start sm:items-center justify-between gap-6">
                                        <div className="flex items-start gap-4">
                                            <div className="p-3 bg-red-100 text-red-600 rounded-xl shrink-0 animate-pulse">
                                                <AlertTriangle className="w-8 h-8" />
                                            </div>
                                            <div>
                                                <h2 className="text-2xl font-black text-slate-900 leading-tight">Threats Detected</h2>
                                                <p className="text-slate-600 font-medium mt-1">
                                                    Target IP <span className="font-mono font-bold text-slate-900 bg-white px-2 py-0.5 rounded border border-slate-200">{ip}</span> is listed on <span className="text-red-600 font-bold">{listedCount} Blacklists</span>.
                                                </p>
                                            </div>
                                        </div>
                                        <div className="text-right hidden sm:block">
                                            <div className="text-sm font-bold text-slate-400 uppercase tracking-wider mb-1">Risk Score</div>
                                            <div className="text-4xl font-black text-red-600">{100 - reputationScore}<span className="text-lg text-slate-300">/100</span></div>
                                        </div>
                                    </div>
                                    {/* Quick List of Failing RBLs */}
                                    <div className="p-6 bg-red-50/20 grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
                                        {listed.map((r, i) => (
                                            <div key={i} className="flex items-center gap-3 bg-white p-3 rounded-lg border border-red-100/50 shadow-sm">
                                                <div className="w-2 h-2 rounded-full bg-red-500 animate-pulse" />
                                                <span className="font-bold text-slate-700 text-sm truncate">{r.name}</span>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            {/* CLEAN HERO - Shows ONLY if clean */}
                            {isClean && (
                                <div className="bg-emerald-50/50 rounded-3xl border border-emerald-100 shadow-lg p-8 text-center relative overflow-hidden group">
                                    <div className="absolute top-0 inset-x-0 h-1 bg-gradient-to-r from-emerald-400 to-teal-400" />
                                    <div className="flex flex-col items-center justify-center gap-4 relative z-10">
                                        <div className="p-4 bg-white shadow-xl shadow-emerald-900/5 rounded-full text-emerald-500 mb-2 group-hover:scale-110 transition-transform duration-300">
                                            <ShieldCheck className="w-12 h-12" />
                                        </div>
                                        <h2 className="text-3xl font-black text-slate-900">All Systems Operational</h2>
                                        <p className="text-emerald-700 font-medium max-w-lg mx-auto">
                                            This IP address is not listed on any of the checked databases. Reputation integrity is 100%.
                                        </p>
                                    </div>
                                </div>
                            )}

                            {/* Scoreboard Metrics */}
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <MetricCard
                                    label="Reputation Score"
                                    value={reputationScore.toString()}
                                    sub="/ 100"
                                    icon={<Activity className="w-5 h-5" />}
                                    trend={reputationScore >= 90 ? "Excellent" : "At Risk"}
                                    color={reputationScore >= 90 ? "emerald" : reputationScore >= 70 ? "amber" : "red"}
                                />
                                <MetricCard
                                    label="Databases Scanned"
                                    value={totalLists.toString()}
                                    sub="RBLs"
                                    icon={<Server className="w-5 h-5" />}
                                    trend="Global Coverage"
                                    color="blue"
                                />
                                <MetricCard
                                    label="Avg. Latency"
                                    value={avgLatency.toString()}
                                    sub="ms"
                                    icon={<Activity className="w-5 h-5" />}
                                    trend="Real-time"
                                    color="purple"
                                />
                            </div>

                            {/* Detailed Dashboard */}
                            <BlacklistDashboard
                                results={results}
                                scannedCount={totalLists}
                                totalLists={totalLists}
                                ip={ip}
                            />
                        </motion.div>
                    )}

                    {loading && (
                        <div className="flex flex-col items-center justify-center py-20 text-center space-y-6">
                            <div className="relative">
                                <div className="w-24 h-24 rounded-full border-4 border-indigo-100 border-t-indigo-600 animate-spin" />
                                <div className="absolute inset-0 flex items-center justify-center">
                                    <Shield className="w-8 h-8 text-indigo-200 animate-pulse" />
                                </div>
                            </div>
                            <div>
                                <h3 className="text-xl font-bold text-slate-900">Auditing Global Blacklists</h3>
                                <p className="text-slate-500 mt-2">Connecting to 50+ DNSBL authorities...</p>
                            </div>
                        </div>
                    )}
                </AnimatePresence>
            </main>


        </div>
    );
}

// Helper for Metrics
const MetricCard = ({ label, value, sub, icon, trend, color }: any) => {
    const colors: any = {
        emerald: "bg-emerald-50 text-emerald-600 border-emerald-100",
        amber: "bg-amber-50 text-amber-600 border-amber-100",
        red: "bg-red-50 text-red-600 border-red-100",
        blue: "bg-blue-50 text-blue-600 border-blue-100",
        purple: "bg-purple-50 text-purple-600 border-purple-100",
    };
    const c = colors[color] || colors.blue;

    return (
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between mb-4">
                <div className={`p-2.5 rounded-xl ${c} bg-opacity-50`}>
                    {icon}
                </div>
                <span className={`text-xs font-bold px-2 py-1 rounded-full uppercase tracking-wider ${c}`}>
                    {trend}
                </span>
            </div>
            <div>
                <div className="flex items-baseline gap-2">
                    <span className="text-4xl font-black text-slate-900 tracking-tight">{value}</span>
                    <span className="text-sm font-bold text-slate-400">{sub}</span>
                </div>
                <div className="text-xs font-bold text-slate-400 uppercase tracking-wider mt-1">{label}</div>
            </div>
        </div>
    );
};
