"use client";

import React from "react";
import { motion } from "framer-motion";
import { Server, Shield, Globe, Terminal, CheckCircle, AlertTriangle, Cloud } from "lucide-react";

export interface CDNResult {
    name: string; // e.g. "Cloudflare", "Direct-to-Origin"
    providerId: string; // internal id for colors 'cloudflare', 'akamai'
    ip: string;
    organization: string; // From IP WHOIS
    headersFound: string[];
    isWafActive: boolean;
    detectedBy: 'Headers' | 'DNS' | 'IP' | 'None';
}

interface CDNDetailsProps {
    result: CDNResult;
}

export const CDNDetails: React.FC<CDNDetailsProps> = ({ result }) => {

    const getProviderStyle = (id: string) => {
        switch (id.toLowerCase()) {
            case 'cloudflare': return { bg: 'bg-[#F38020]/10', text: 'text-[#F38020]', border: 'border-[#F38020]/20' };
            case 'akamai': return { bg: 'bg-[#0099CC]/10', text: 'text-[#0099CC]', border: 'border-[#0099CC]/20' };
            case 'fastly': return { bg: 'bg-[#EB2C2E]/10', text: 'text-[#EB2C2E]', border: 'border-[#EB2C2E]/20' };
            case 'cloudfront': return { bg: 'bg-[#FF9900]/10', text: 'text-[#FF9900]', border: 'border-[#FF9900]/20' };
            case 'azure': return { bg: 'bg-[#0078D4]/10', text: 'text-[#0078D4]', border: 'border-[#0078D4]/20' };
            case 'vercel': return { bg: 'bg-black/5', text: 'text-black', border: 'border-black/10' };
            case 'netlify': return { bg: 'bg-[#00C7B7]/10', text: 'text-[#00C7B7]', border: 'border-[#00C7B7]/20' };
            default: return { bg: 'bg-slate-50', text: 'text-slate-900', border: 'border-slate-200' };
        }
    };

    const style = getProviderStyle(result.providerId);
    const isDirect = result.providerId === 'direct';

    return (
        <div className="space-y-6">

            {/* Hero Result Card */}
            <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                className={`rounded-3xl p-8 border ${style.border} ${style.bg} relative overflow-hidden`}
            >
                {/* Background Pattern */}
                <div className="absolute top-0 right-0 p-12 opacity-10 transform translate-x-1/4 -translate-y-1/4">
                    <Cloud className={`w-64 h-64 ${style.text}`} />
                </div>

                <div className="relative z-10 flex flex-col md:flex-row items-center gap-6 text-center md:text-left">
                    <div className={`p-4 rounded-2xl bg-white shadow-lg ${style.text}`}>
                        {isDirect ? <Server className="w-12 h-12" /> : <Cloud className="w-12 h-12" />}
                    </div>

                    <div className="flex-1">
                        <p className={`text-sm font-bold uppercase tracking-widest opacity-70 mb-1 ${isDirect ? 'text-slate-600' : style.text}`}>
                            {isDirect ? "Network Status" : "CDN Detected"}
                        </p>
                        <h2 className="text-3xl md:text-4xl font-black text-slate-900 mb-2">
                            {result.name}
                        </h2>
                        <div className="flex items-center justify-center md:justify-start gap-3">
                            {result.isWafActive && (
                                <span className="inline-flex items-center gap-1.5 px-3 py-1 rounded-full bg-emerald-100 text-emerald-700 text-xs font-bold border border-emerald-200">
                                    <Shield className="w-3 h-3" /> WAF Active
                                </span>
                            )}
                            {!isDirect && (
                                <span className={`inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-xs font-bold border bg-white/50 ${style.text} ${style.border}`}>
                                    <CheckCircle className="w-3 h-3" /> Verified via {result.detectedBy}
                                </span>
                            )}
                        </div>
                    </div>
                </div>
            </motion.div>

            {/* Technical Proof Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">

                {/* Headers Fingerprint */}
                <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-sm">
                    <div className="flex items-center gap-2 mb-4">
                        <Terminal className="w-5 h-5 text-slate-400" />
                        <h3 className="font-bold text-slate-900 text-sm uppercase tracking-wider">Fingerprint Headers</h3>
                    </div>

                    {result.headersFound.length > 0 ? (
                        <div className="bg-slate-900 rounded-xl p-4 overflow-x-auto">
                            <ul className="space-y-2">
                                {result.headersFound.map((h, i) => (
                                    <li key={i} className="font-mono text-xs text-emerald-400 flex items-start gap-2">
                                        <span className="text-slate-500 select-none">{'>'}</span>
                                        {h}
                                    </li>
                                ))}
                            </ul>
                        </div>
                    ) : (
                        <div className="bg-slate-50 rounded-xl p-4 text-center text-slate-400 text-sm font-medium italic">
                            No specific CDN headers detected.
                        </div>
                    )}
                </div>

                {/* Network Identity */}
                <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-sm flex flex-col justify-between">
                    <div>
                        <div className="flex items-center gap-2 mb-4">
                            <Globe className="w-5 h-5 text-slate-400" />
                            <h3 className="font-bold text-slate-900 text-sm uppercase tracking-wider">Edge Identity</h3>
                        </div>

                        <div className="space-y-4">
                            <div>
                                <p className="text-xs text-slate-400 font-bold mb-1">Organization / ASN</p>
                                <p className="text-lg font-bold text-slate-900">{result.organization || "Unknown"}</p>
                            </div>
                            <div>
                                <p className="text-xs text-slate-400 font-bold mb-1">IP Address</p>
                                <p className="font-mono text-sm text-slate-600 bg-slate-50 inline-block px-2 py-1 rounded border border-slate-100">
                                    {result.ip}
                                </p>
                            </div>
                        </div>
                    </div>

                    <div className="mt-6 pt-6 border-t border-slate-100">
                        <div className="flex items-center gap-2 text-xs text-slate-500 font-medium">
                            {result.detectedBy === 'DNS' && (
                                <>
                                    <CheckCircle className="w-4 h-4 text-emerald-500" />
                                    Confirmed via Nameservers
                                </>
                            )}
                            {result.detectedBy === 'IP' && (
                                <>
                                    <CheckCircle className="w-4 h-4 text-emerald-500" />
                                    Identified via IP Ownership
                                </>
                            )}
                            {result.detectedBy === 'Headers' && (
                                <>
                                    <CheckCircle className="w-4 h-4 text-emerald-500" />
                                    Identified via Response Headers
                                </>
                            )}
                        </div>
                    </div>
                </div>

            </div>
        </div>
    );
};
