"use client";

import React from "react";
import { motion } from "framer-motion";
import { ShieldCheck, ShieldAlert, Key, Lock, Copy } from "lucide-react";

export interface DKIMRecord {
    raw: string;
    version: string;
    keyType: string;
    publicKey: string;
    keyLength?: number;
    flags?: string;
    isKeyValid: boolean;
}

interface DKIMKeyDetailsProps {
    data: DKIMRecord;
}

export const DKIMKeyDetails: React.FC<DKIMKeyDetailsProps> = ({ data }) => {

    // Security check logic
    const isWeakKey = (data.keyType === 'RSA' && (data.keyLength || 0) < 2048);

    const copyToClipboard = (text: string) => {
        navigator.clipboard.writeText(text);
    };

    return (
        <div className="space-y-6">

            {/* Security Advisor */}
            {isWeakKey ? (
                <div className="bg-amber-50 border border-amber-200 rounded-2xl p-6 flex items-start gap-4">
                    <ShieldAlert className="w-6 h-6 text-amber-600 flex-shrink-0 mt-1" />
                    <div>
                        <h4 className="font-bold text-amber-900 text-lg mb-1">Upgrade Recommended: Weak Key Detected</h4>
                        <p className="text-amber-800/80 text-sm leading-relaxed">
                            This domain uses a <strong>{data.keyLength}-bit RSA key</strong>.
                            1024-bit keys are considered cryptographically weak by modern standards and may be vulnerable to spoofing.
                            Please upgrade to a <strong>2048-bit</strong> key.
                        </p>
                    </div>
                </div>
            ) : (
                <div className="bg-emerald-50 border border-emerald-200 rounded-2xl p-6 flex items-start gap-4">
                    <ShieldCheck className="w-6 h-6 text-emerald-600 flex-shrink-0 mt-1" />
                    <div>
                        <h4 className="font-bold text-emerald-900 text-lg mb-1">Cryptographically Sound</h4>
                        <p className="text-emerald-800/80 text-sm leading-relaxed">
                            This valid <strong>{data.keyLength}-bit {data.keyType}</strong> key meets current industry security standards.
                        </p>
                    </div>
                </div>
            )}

            {/* Key Specs */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="bg-white p-5 rounded-xl border border-slate-200 shadow-sm">
                    <span className="text-xs font-bold text-slate-400 uppercase tracking-wider block mb-1">Algorithm</span>
                    <div className="flex items-center gap-2">
                        <Lock className="w-4 h-4 text-blue-500" />
                        <span className="text-lg font-bold text-slate-900">{data.keyType || "RSA"}</span>
                    </div>
                </div>
                <div className="bg-white p-5 rounded-xl border border-slate-200 shadow-sm">
                    <span className="text-xs font-bold text-slate-400 uppercase tracking-wider block mb-1">Key Size</span>
                    <div className="flex items-center gap-2">
                        <Key className="w-4 h-4 text-blue-500" />
                        <span className="text-lg font-bold text-slate-900">{data.keyLength ? `${data.keyLength} bits` : "Unknown"}</span>
                    </div>
                </div>
                <div className="bg-white p-5 rounded-xl border border-slate-200 shadow-sm">
                    <span className="text-xs font-bold text-slate-400 uppercase tracking-wider block mb-1">Status</span>
                    <div className="flex items-center gap-2">
                        <div className={`w-2.5 h-2.5 rounded-full ${data.isKeyValid ? 'bg-emerald-500' : 'bg-red-500'}`} />
                        <span className="text-lg font-bold text-slate-900">{data.isKeyValid ? "Valid Syntax" : "Invalid Syntax"}</span>
                    </div>
                </div>
            </div>

            {/* Public Key Block */}
            <div className="bg-white rounded-2xl border border-slate-200 overflow-hidden shadow-sm">
                <div className="bg-slate-50 px-6 py-4 border-b border-slate-200 flex justify-between items-center">
                    <h3 className="font-bold text-slate-700 text-sm uppercase tracking-wide">Public Key Data (p=)</h3>
                    <button
                        onClick={() => copyToClipboard(data.publicKey)}
                        className="text-xs font-bold text-blue-600 hover:text-blue-700 flex items-center gap-1 active:scale-95 transition-transform"
                    >
                        <Copy className="w-3.5 h-3.5" /> Copy Key
                    </button>
                </div>
                <div className="p-6 bg-slate-50/30">
                    <div className="relative font-mono text-sm leading-relaxed text-slate-600 break-all p-4 bg-white border border-blue-100 rounded-lg shadow-inner">
                        {data.publicKey}
                    </div>
                </div>
            </div>

            {/* Raw Record */}
            <div className="opacity-80">
                <div className="text-xs font-bold text-slate-400 uppercase tracking-wider mb-2">Raw DNS Record</div>
                <div className="font-mono text-xs bg-slate-900 text-slate-300 p-4 rounded-xl break-all">
                    {data.raw}
                </div>
            </div>

        </div>
    );
};
