"use client";

import React from "react";
import { motion } from "framer-motion";
import { Shield, AlertTriangle, CheckCircle, Smartphone, Mail, Info } from "lucide-react";

export interface DmarcTag {
    tag: string;
    value: string;
    description: string;
    status: 'good' | 'warning' | 'danger' | 'neutral';
}

interface DMARCDeconstructorProps {
    record: string;
}

export const DMARCDeconstructor: React.FC<DMARCDeconstructorProps> = ({ record }) => {
    // 1. Parse the Record
    // Example: v=DMARC1; p=reject; rua=mailto:dmarc@example.com
    const tags = record.split(';').map(t => t.trim()).filter(t => t);

    // Helper to identify tag meanings
    const analyzeTag = (rawTag: string): DmarcTag => {
        const [key, ...values] = rawTag.split('=');
        if (!key || !values) return { tag: rawTag, value: '', description: 'Invalid Syntax', status: 'danger' };

        const tagKey = key.trim().toLowerCase();
        const value = values.join('=').trim(); // Rejoin in case value contains = (weird but possible)

        switch (tagKey) {
            case 'v':
                return {
                    tag: 'Version',
                    value,
                    description: 'Identifies this as a DMARC record. Must be "DMARC1".',
                    status: value === 'DMARC1' ? 'good' : 'danger'
                };
            case 'p':
                if (value === 'reject') return { tag: 'Policy', value, description: 'Strict Protection: Reject failing emails.', status: 'good' };
                if (value === 'quarantine') return { tag: 'Policy', value, description: 'Moderate Protection: Send failing emails to Spam.', status: 'warning' };
                return { tag: 'Policy', value, description: 'No Protection: Monitoring mode only.', status: 'danger' };
            case 'rua':
                return { tag: 'Aggregate Reports', value, description: 'Where to send daily aggregate reports.', status: 'good' };
            case 'ruf':
                return { tag: 'Forensic Reports', value, description: 'Where to send detailed failure reports.', status: 'neutral' };
            case 'pct':
                const pctVal = parseInt(value);
                if (pctVal === 100) return { tag: 'Percentage', value: `${value}%`, description: 'Policy applies to 100% of emails.', status: 'good' };
                return { tag: 'Percentage', value: `${value}%`, description: `Policy only applies to ${value}% of emails (Testing).`, status: 'warning' };
            case 'aspf':
            case 'adkim':
                if (value === 's') return { tag: 'Alignment', value: 'Strict', description: `Strict ${tagKey === 'aspf' ? 'SPF' : 'DKIM'} alignment required.`, status: 'good' };
                return { tag: 'Alignment', value: 'Relaxed', description: `Relaxed ${tagKey === 'aspf' ? 'SPF' : 'DKIM'} alignment allowed (Standard).`, status: 'neutral' };
            case 'sp':
                if (value === 'reject') return { tag: 'Subdomain Policy', value, description: 'Subdomains: Reject failing emails.', status: 'good' };
                return { tag: 'Subdomain Policy', value, description: `Subdomains use policy: ${value}`, status: value === 'none' ? 'danger' : 'warning' };
            default:
                return { tag: tagKey, value, description: 'Optional configuration tag.', status: 'neutral' };
        }
    };

    const parsedTags = tags.map(analyzeTag);

    // Color Helpers
    const getStatusColor = (status: DmarcTag['status']) => {
        switch (status) {
            case 'good': return "bg-emerald-500";
            case 'warning': return "bg-amber-500";
            case 'danger': return "bg-red-500";
            default: return "bg-slate-300";
        }
    };

    const getStatusBg = (status: DmarcTag['status']) => {
        switch (status) {
            case 'good': return "bg-emerald-50 border-emerald-100";
            case 'warning': return "bg-amber-50 border-amber-100";
            case 'danger': return "bg-red-50 border-red-100";
            default: return "bg-slate-50 border-slate-100";
        }
    };

    return (
        <div className="space-y-8">

            {/* Raw Record Box */}
            <div className="bg-slate-900 rounded-2xl p-6 shadow-xl relative overflow-hidden group">
                <div className="absolute top-0 right-0 p-4 opacity-10">
                    <Shield className="w-16 h-16 text-white" />
                </div>
                <h3 className="text-slate-400 text-xs font-bold uppercase tracking-widest mb-3">Raw Record</h3>
                <div className="font-mono text-emerald-400 text-sm md:text-base break-all bg-slate-800/50 p-4 rounded-xl border border-slate-700/50">
                    {record}
                </div>
            </div>

            {/* Deconstructed Grid */}
            <div className="grid grid-cols-1 md:grid-cols-1 gap-4">
                {parsedTags.map((item, index) => (
                    <motion.div
                        key={index}
                        initial={{ opacity: 0, x: -10 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ delay: index * 0.1 }}
                        className={`relative overflow-hidden rounded-xl border p-5 flex flex-col md:flex-row md:items-center gap-4 ${getStatusBg(item.status)}`}
                    >
                        {/* Traffic Light Bar */}
                        <div className={`absolute left-0 top-0 bottom-0 w-1.5 ${getStatusColor(item.status)}`} />

                        <div className="flex-1">
                            <div className="flex items-center gap-3 mb-1">
                                <span className="font-mono font-bold text-slate-900 text-lg">{item.tag.toUpperCase()}</span>
                                <span className="text-slate-400 font-mono text-sm">=</span>
                                <span className="font-mono font-bold text-blue-700 bg-blue-50 px-2 py-0.5 rounded">{item.value}</span>
                            </div>
                            <p className="text-slate-600 font-medium text-sm pl-[2px]">{item.description}</p>
                        </div>

                        <div className="flex items-center gap-2 md:ml-auto">
                            {item.status === 'good' && <div className="flex items-center gap-1.5 px-3 py-1 rounded-full bg-white border border-emerald-200 text-emerald-700 text-xs font-bold uppercase tracking-wider"><CheckCircle className="w-3.5 h-3.5" /> Secure</div>}
                            {item.status === 'warning' && <div className="flex items-center gap-1.5 px-3 py-1 rounded-full bg-white border border-amber-200 text-amber-700 text-xs font-bold uppercase tracking-wider"><AlertTriangle className="w-3.5 h-3.5" /> Weak</div>}
                            {item.status === 'danger' && <div className="flex items-center gap-1.5 px-3 py-1 rounded-full bg-white border border-red-200 text-red-700 text-xs font-bold uppercase tracking-wider"><Shield className="w-3.5 h-3.5" /> Risk</div>}
                        </div>

                    </motion.div>
                ))}
            </div>

        </div>
    );
};
