"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Search, Loader2, Globe } from "lucide-react";

import { WorldMap, MapNode } from "@/components/dns-propagation/WorldMap";
import { PropagationGrid } from "@/components/dns-propagation/PropagationGrid";
import { checkDnsPropagation } from "@/app/actions/dns";

const RECORD_TYPES = ["A", "AAAA", "CNAME", "MX", "NS", "TXT"];

export default function DnsPropagationClient() {
    const [domain, setDomain] = useState("");
    const [type, setType] = useState("A");
    const [isSearching, setIsSearching] = useState(false);

    // Initial State: Pending nodes distributed globally
    const [nodes, setNodes] = useState<MapNode[]>([
        { id: "US-WC", name: "Google DNS (California, USA)", status: 'pending', x: 20, y: 35 },
        { id: "Global", name: "Cloudflare (Global Anycast)", status: 'pending', x: 48, y: 40 }, // Atlantic
        { id: "CN", name: "AliDNS (Shanghai, China)", status: 'pending', x: 75, y: 38 },
        { id: "EU", name: "Quad9 (Zurich, Switzerland)", status: 'pending', x: 53, y: 32 },
        { id: "CY", name: "AdGuard (Nicosia, Cyprus)", status: 'pending', x: 58, y: 40 },
        { id: "Local", name: "Local Resolver (Your Node)", status: 'pending', x: 15, y: 70 }, // South America/Mock
    ]);

    const handleCheck = async () => {
        if (!domain) return;
        setIsSearching(true);

        // Reset status to pending
        setNodes(prev => prev.map(n => ({ ...n, status: 'pending', ip: undefined })));

        try {
            const results = await checkDnsPropagation(domain, type);

            // Map results to nodes
            // Note: The order in actions/dns.ts is Google, Cloudflare, Ali, Quad9, AdGuard, Local
            // We map them by index or provider name matching

            const newNodes = [...nodes];

            const updateNode = (id: string, result: any) => {
                const nodeIdx = newNodes.findIndex(n => n.id === id);
                if (nodeIdx !== -1 && result) {
                    newNodes[nodeIdx] = {
                        ...newNodes[nodeIdx],
                        status: result.status === 'ok' ? 'resolved' : 'error',
                        ip: result.value ? result.value.split(',')[0] : undefined // value might be list
                    };
                }
            };

            // Mapping Logic relies on provider strings from actions/dns.ts
            const google = results.find(r => r.provider.includes("Google"));
            updateNode("US-WC", google);

            const cf = results.find(r => r.provider.includes("Cloudflare"));
            updateNode("Global", cf);

            const ali = results.find(r => r.provider.includes("AliDNS"));
            updateNode("CN", ali);

            const quad9 = results.find(r => r.provider.includes("Quad9"));
            updateNode("EU", quad9);

            const adguard = results.find(r => r.provider.includes("AdGuard"));
            updateNode("CY", adguard);

            const local = results.find(r => r.provider.includes("Local"));
            updateNode("Local", local);

            setNodes(newNodes);

        } catch (error) {
            console.error(error);
        } finally {
            setIsSearching(false);
        }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">
            {/* Header */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                    >
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-4 font-jakarta">
                            Global DNS Propagation
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto">
                            Check your DNS records against multiple global nodes instantly.
                        </p>
                    </motion.div>
                </div>
            </section>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Bar */}
                <div className="bg-white p-2 rounded-2xl shadow-sm border border-[#E2E8F0] flex flex-col sm:flex-row gap-2">
                    <div className="relative min-w-[120px]">
                        <select
                            value={type}
                            onChange={(e) => setType(e.target.value)}
                            className="w-full h-12 pl-4 pr-8 appearance-none bg-slate-50 rounded-xl font-bold text-slate-700 outline-none focus:ring-2 focus:ring-blue-100"
                        >
                            {RECORD_TYPES.map(t => <option key={t} value={t}>{t}</option>)}
                        </select>
                    </div>

                    <div className="flex-1">
                        <input
                            type="text"
                            placeholder="example.com"
                            value={domain}
                            onChange={(e) => setDomain(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleCheck()}
                            className="w-full h-12 px-4 bg-transparent text-lg font-medium outline-none placeholder-slate-400"
                        />
                    </div>

                    <button
                        onClick={handleCheck}
                        disabled={isSearching || !domain}
                        className="h-12 px-8 bg-[#2563EB] hover:bg-[#1E3A8A] text-white font-bold rounded-xl transition-all shadow-md active:scale-95 disabled:opacity-70 flex items-center justify-center gap-2"
                    >
                        {isSearching ? <Loader2 className="w-5 h-5 animate-spin" /> : <Globe className="w-5 h-5" />}
                        Check Status
                    </button>
                </div>

                {/* Dashboard Split View */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Left: Results List */}
                    <div className="lg:col-span-1">
                        <PropagationGrid nodes={nodes} />
                    </div>

                    {/* Right: Map */}
                    <div className="lg:col-span-2">
                        <WorldMap nodes={nodes} />
                    </div>
                </div>

            </main>


        </div>
    );
}
