"use client";

import React from "react";
import { motion } from "framer-motion";
import { Shield, ShieldAlert, ShieldCheck, Lock, Unlock, Key, CheckCircle, XCircle, ArrowDown } from "lucide-react";

export interface TrustNode {
    level: string; // "Root (.)", "TLD (.com)", "Domain (example.com)"
    status: 'secure' | 'insecure' | 'bogus';
    keys: {
        id: number;
        algorithm: string;
        type: 'KSK' | 'ZSK' | 'DS';
        flags?: number;
    }[];
    message: string;
}

interface DNSSECChainProps {
    chain: TrustNode[];
}

export const DNSSECChain: React.FC<DNSSECChainProps> = ({ chain }) => {

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'secure': return 'bg-emerald-50 border-emerald-200 text-emerald-900';
            case 'bogus': return 'bg-red-50 border-red-200 text-red-900';
            default: return 'bg-slate-50 border-slate-200 text-slate-800';
        }
    };

    const getIcon = (status: string) => {
        switch (status) {
            case 'secure': return <Lock className="w-5 h-5 text-emerald-600" />;
            case 'bogus': return <ShieldAlert className="w-5 h-5 text-red-600" />;
            default: return <Unlock className="w-5 h-5 text-slate-400" />;
        }
    };

    return (
        <div className="max-w-2xl mx-auto py-8">
            <h3 className="text-center font-bold text-slate-900 text-xl mb-8">Chain of Trust</h3>

            <div className="space-y-0 relative">
                {chain.map((node, index) => {
                    const isLast = index === chain.length - 1;

                    return (
                        <div key={index} className="relative z-10">
                            {/* Card */}
                            <motion.div
                                initial={{ opacity: 0, y: 20 }}
                                animate={{ opacity: 1, y: 0 }}
                                transition={{ delay: index * 0.2 }}
                                className={`rounded-2xl border p-6 shadow-sm relative z-20 ${getStatusColor(node.status)}`}
                            >
                                <div className="flex items-center justify-between mb-4">
                                    <div className="flex items-center gap-3">
                                        <div className={`p-2 rounded-lg bg-white/50 border border-black/5`}>
                                            {getIcon(node.status)}
                                        </div>
                                        <div>
                                            <h4 className="font-bold text-lg">{node.level}</h4>
                                            <p className="text-xs font-bold opacity-60 uppercase tracking-widest">{node.status}</p>
                                        </div>
                                    </div>
                                    {node.status === 'secure' && <CheckCircle className="w-6 h-6 text-emerald-500/50" />}
                                    {node.status === 'bogus' && <XCircle className="w-6 h-6 text-red-500/50" />}
                                </div>

                                <div className="space-y-3 bg-white/60 rounded-xl p-4 border border-black/5">
                                    {node.keys.length > 0 ? node.keys.map((k, i) => (
                                        <div key={i} className="flex items-center justify-between text-sm">
                                            <div className="flex items-center gap-2">
                                                <Key className="w-3.5 h-3.5 opacity-40" />
                                                <span className="font-mono font-bold text-blue-700">{k.type}</span>
                                                <span className="font-mono text-xs opacity-70">Tag: {k.id}</span>
                                            </div>
                                            <span className="text-xs opacity-60 font-medium">{k.algorithm}</span>
                                        </div>
                                    )) : (
                                        <p className="text-sm italic opacity-60">{node.message || "No keys found."}</p>
                                    )}
                                </div>
                            </motion.div>

                            {/* Connector Line */}
                            {!isLast && (
                                <div className="h-10 w-0.5 bg-slate-200 mx-auto my-[-4px] relative z-0 flex items-center justify-center">
                                    {node.status === 'secure' && chain[index + 1].status !== 'bogus' ? (
                                        <div className="w-1 h-full bg-emerald-200" />
                                    ) : (
                                        <div className="w-0.5 h-full border-r-2 border-dashed border-slate-300" />
                                    )}
                                </div>
                            )}
                        </div>
                    );
                })}
            </div>

            <div className="mt-8 text-center">
                <p className="text-sm text-slate-400">
                    * This tool verifies the presence of DNSKEY and DS records to validate the chain of trust.
                </p>
            </div>
        </div>
    );
};
