"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Clock, Loader2, Search } from "lucide-react";

import { DomainTimelineCard, DomainTimeline } from "@/components/domain-expiry/DomainTimelineCard";

export default function DomainExpiryClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [result, setResult] = useState<DomainTimeline | null>(null);
    const [error, setError] = useState("");

    const handleCheck = async () => {
        if (!domain) return;
        setLoading(true);
        setError("");
        setResult(null);

        try {
            const res = await fetch('/api/domain/expiry', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to check domain expiry");
            }
            setResult(json);

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-indigo-600 rounded-lg shadow-lg shadow-indigo-600/20 text-white">
                                <Clock className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                Domain Age & <span className="text-indigo-600">Expiry Tracker</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Monitor registration dates, expiration countdowns, and exact domain age for any TLD.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-12">

                {/* Search Box */}
                <div className="max-w-xl mx-auto relative z-10">
                    <motion.div
                        initial={{ scale: 0.95, opacity: 0 }}
                        animate={{ scale: 1, opacity: 1 }}
                        className="bg-white rounded-full p-2 shadow-2xl shadow-indigo-500/10 border border-slate-100 flex items-center gap-2 focus-within:ring-4 focus-within:ring-indigo-500/10 focus-within:border-indigo-500/50 transition-all duration-300"
                    >
                        <div className="flex-1 relative group pl-4">
                            <Search className="absolute left-0 top-1/2 -translate-y-1/2 w-6 h-6 text-slate-300 group-focus-within:text-indigo-500 transition-colors duration-300" />
                            <input
                                type="text"
                                placeholder="example.com"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                onKeyDown={(e) => e.key === 'Enter' && handleCheck()}
                                className="w-full h-14 pl-10 pr-4 bg-transparent border-none font-bold text-xl text-slate-800 placeholder:text-slate-300 outline-none font-jakarta"
                                suppressHydrationWarning
                                autoFocus
                            />
                        </div>

                        <motion.button
                            whileHover={{ scale: 1.05 }}
                            whileTap={{ scale: 0.95 }}
                            onClick={handleCheck}
                            disabled={loading || !domain}
                            className="h-12 px-8 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-full transition-colors shadow-lg shadow-indigo-600/30 flex items-center justify-center gap-2 whitespace-nowrap"
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : "Check Timeline"}
                        </motion.button>
                    </motion.div>
                </div>

                {error && (
                    <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="max-w-2xl mx-auto bg-red-50 border border-red-100 text-red-600 p-4 rounded-xl text-center font-medium">
                        {error}
                    </motion.div>
                )}

                <AnimatePresence mode="wait">
                    {result && !loading && (
                        <motion.div
                            key="result"
                            initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ duration: 0.4 }}
                        >
                            <DomainTimelineCard data={result} />
                        </motion.div>
                    )}
                </AnimatePresence>
            </main>


        </div>
    );
}
