"use client";

import React from "react";
import { motion } from "framer-motion";
import { Clock, Calendar, AlertTriangle, CheckCircle, RefreshCw, Activity } from "lucide-react";

export interface DomainTimeline {
    domain: string;
    createdDate: string;
    expiryDate: string;
    updatedDate: string;
    ageYears: number;
    daysToExpiry: number;
    status: 'Active' | 'Grace Period' | 'Redemption' | 'Expired';
    registrar: string;
}

interface DomainTimelineCardProps {
    data: DomainTimeline;
}

export const DomainTimelineCard: React.FC<DomainTimelineCardProps> = ({ data }) => {

    // Calculate Progress % for the Gauge
    // Total lifespan so far + time left
    // Actually, for "Time Remaining" gauge, usually it's % of the current 1 year cycle or total age?
    // Let's do % of Life Spent vs Left? Or just a "Health" meter?
    // User asked for "percentage of time remaining in the current registration period".
    // Usually domains are 1 year. But let's approximate based on (Now - Created) / (Expiry - Created).
    const created = new Date(data.createdDate).getTime();
    const expiry = new Date(data.expiryDate).getTime();
    const now = Date.now();
    const totalDuration = expiry - created;
    const elapsed = now - created;
    let progress = Math.min(100, Math.max(0, (elapsed / totalDuration) * 100));

    // If expired, progress is 100
    if (data.daysToExpiry < 0) progress = 100;

    // Format Dates
    const formatDate = (d: string) => new Date(d).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });

    // Status Color
    const getStatusColor = () => {
        if (data.daysToExpiry < 0) return "text-red-600 bg-red-50 border-red-100";
        if (data.daysToExpiry < 30) return "text-amber-600 bg-amber-50 border-amber-100";
        return "text-emerald-600 bg-emerald-50 border-emerald-100";
    };

    return (
        <div className="bg-white border border-slate-200 rounded-3xl p-8 shadow-xl shadow-blue-500/5 relative overflow-hidden">

            {/* Header */}
            <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-6 mb-10">
                <div>
                    <h2 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">{data.domain}</h2>
                    <div className="flex items-center gap-2 mt-1">
                        <span className="text-slate-500 font-medium text-sm">Registrar:</span>
                        <span className="font-bold text-slate-700 text-sm bg-slate-100 px-2 py-0.5 rounded-full">{data.registrar}</span>
                    </div>
                </div>

                <div className={`px-4 py-2 rounded-xl border flex items-center gap-2 font-bold text-sm ${getStatusColor()}`}>
                    {data.daysToExpiry < 0 ? <AlertTriangle className="w-4 h-4" /> : <CheckCircle className="w-4 h-4" />}
                    {data.status.toUpperCase()}
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-12 items-center">

                {/* 1. Circular Countdown Gauge */}
                <div className="flex flex-col items-center justify-center relative py-6">
                    <div className="relative w-64 h-64">
                        {/* Glow Filter */}
                        <svg className="absolute inset-0 w-full h-full transform -rotate-90 filter drop-shadow-[0_0_15px_rgba(59,130,246,0.5)]">
                            {/* Background Track */}
                            <circle
                                cx="128" cy="128" r="100"
                                stroke="#F1F5F9" strokeWidth="20" fill="transparent"
                                strokeLinecap="round"
                            />
                            {/* Progress Arc */}
                            <circle
                                cx="128" cy="128" r="100"
                                stroke="currentColor" strokeWidth="20" fill="transparent"
                                strokeDasharray={2 * Math.PI * 100}
                                strokeDashoffset={2 * Math.PI * 100 * (1 - (progress) / 100)}
                                className={`transition-all duration-1000 ease-out ${data.daysToExpiry < 30 ? "text-amber-500" : "text-blue-600"}`}
                                strokeLinecap="round"
                            />
                        </svg>
                        <div className="absolute inset-0 flex flex-col items-center justify-center">
                            <motion.div
                                initial={{ scale: 0.5, opacity: 0 }}
                                animate={{ scale: 1, opacity: 1 }}
                                transition={{ type: "spring", stiffness: 100 }}
                                className={`text-7xl font-black tracking-tighter ${data.daysToExpiry < 30 ? "text-amber-600" : "text-slate-900"}`}
                            >
                                {data.daysToExpiry}
                            </motion.div>
                            <div className="text-xs font-bold text-slate-400 uppercase tracking-widest mt-2">Days Left</div>
                        </div>
                    </div>
                </div>

                {/* 2. Key Metrics Grid */}
                <div className="lg:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-6">
                    <MetricBox
                        label="Total Domain Age"
                        value={`${data.ageYears.toFixed(1)} Years`}
                        icon={<Clock className="w-5 h-5 text-blue-500" />}
                        desc={`Registered on ${formatDate(data.createdDate)}`}
                    />
                    <MetricBox
                        label="Expiration Date"
                        value={formatDate(data.expiryDate)}
                        icon={<Calendar className="w-5 h-5 text-indigo-500" />}
                        desc="Ensure auto-renew is ON"
                        highlight={data.daysToExpiry < 30}
                    />
                    <MetricBox
                        label="Registration Status"
                        value={data.status}
                        icon={<Activity className="w-5 h-5 text-emerald-500" />}
                        desc="Current Registry State"
                    />
                    <MetricBox
                        label="Last Updated"
                        value={formatDate(data.updatedDate)}
                        icon={<RefreshCw className="w-5 h-5 text-slate-500" />}
                        desc="Recent WHOIS Change"
                    />
                </div>
            </div>

            {/* 3. Horizontal Timeline */}
            <div className="mt-20 pt-0 relative px-4">
                <div className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-10 pl-2">Lifecycle Timeline</div>

                {/* Track */}
                <div className="relative h-4 bg-slate-100 rounded-full w-full overflow-visible">
                    <motion.div
                        initial={{ width: 0 }}
                        animate={{ width: `${progress}%` }}
                        transition={{ duration: 1.5, ease: "easeOut" }}
                        className="absolute top-0 left-0 h-full bg-gradient-to-r from-blue-400 to-blue-600 rounded-full shadow-[0_0_20px_rgba(37,99,235,0.4)]"
                    />

                    {/* Markers Container - Absolute to full width */}
                    <div className="absolute top-0 left-0 w-full h-full pointer-events-none">
                        <TimelineMarker pos={0} label="Registered" date={formatDate(data.createdDate)} active />
                        <TimelineMarker pos={progress} label="Today" date="Now" active pulse isCurrent />
                        <TimelineMarker pos={100} label="Expires" date={formatDate(data.expiryDate)} alignRight />
                    </div>
                </div>
            </div>

        </div>
    );
};

const MetricBox = ({ label, value, icon, desc, highlight }: any) => (
    <motion.div
        whileHover={{ y: -5, boxShadow: "0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04)" }}
        className={`p-6 rounded-3xl border transition-all duration-300 ${highlight ? "bg-amber-50 border-amber-200" : "bg-white border-slate-100 hover:border-blue-100/50"}`}
    >
        <div className="flex items-center gap-3 mb-4">
            <div className={`p-3 rounded-2xl ${highlight ? "bg-amber-100/80" : "bg-blue-50/50"}`}>
                {icon}
            </div>
            <span className={`text-[11px] font-bold uppercase tracking-widest ${highlight ? "text-amber-700" : "text-slate-500"}`}>{label}</span>
        </div>
        <div className={`text-3xl font-black tracking-tight ${highlight ? "text-amber-900" : "text-slate-900"}`}>{value}</div>
        <div className={`text-xs mt-2 font-medium ${highlight ? "text-amber-600" : "text-slate-400"}`}>{desc}</div>
    </motion.div>
);

const TimelineMarker = ({ pos, label, date, active, pulse, alignRight, isCurrent }: any) => (
    <div
        className={`absolute top-1/2 -translate-y-1/2 z-10 flex flex-col ${alignRight ? "items-end" : "items-center"}`}
        style={{ left: `${pos}%`, transform: `translateX(${alignRight ? '-100%' : isCurrent ? '-50%' : '0%'})` }}
    >
        {/* Dot on the line */}
        <div className={`w-6 h-6 rounded-full border-[4px] border-white shadow-lg transition-transform hover:scale-125
            ${active ? "bg-blue-600" : "bg-slate-300"} 
            ${pulse ? "animate-pulse ring-4 ring-blue-500/30" : ""}`}
        />

        {/* Label below line */}
        <div className={`absolute top-8 ${alignRight ? "text-right" : isCurrent ? "text-center" : "text-left"} min-w-[100px]`}>
            <div className={`text-xs font-bold ${active ? "text-blue-600" : "text-slate-500"}`}>{label}</div>
            <div className="text-[10px] font-mono text-slate-400 mt-1">{date}</div>
        </div>
    </div>
);
