"use client";

import React from "react";
import { motion } from "framer-motion";
import { Shield, Server, AlertTriangle, CheckCircle, Lightbulb, Ban } from "lucide-react";

export interface HealthCheck {
    id: string;
    name: string;
    status: 'pass' | 'warn' | 'fail';
    message: string;
}

export interface EmailHealthReport {
    score: number;
    checks: HealthCheck[];
    recommendations: string[];
}

interface EmailHealthDashboardProps {
    report: EmailHealthReport;
}

export const EmailHealthDashboard: React.FC<EmailHealthDashboardProps> = ({ report }) => {

    // Group checks
    const authChecks = report.checks.filter(c => ['spf', 'dmarc', 'dkim'].includes(c.id));
    const mxCheck = report.checks.find(c => c.id === 'mx');
    const blacklistCheck = report.checks.find(c => c.id === 'blacklist');

    const getStatusColor = (status: HealthCheck['status']) => {
        switch (status) {
            case 'pass': return "bg-emerald-50 border-emerald-100 text-emerald-900";
            case 'warn': return "bg-amber-50 border-amber-100 text-amber-900";
            case 'fail': return "bg-red-50 border-red-100 text-red-900";
            default: return "bg-slate-50 border-slate-100";
        }
    };

    const getIcon = (status: HealthCheck['status']) => {
        switch (status) {
            case 'pass': return <CheckCircle className="w-5 h-5 text-emerald-600" />;
            case 'warn': return <AlertTriangle className="w-5 h-5 text-amber-600" />;
            case 'fail': return <AlertTriangle className="w-5 h-5 text-red-600" />;
        }
    };

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">

            {/* 1. Authentication Module (SPF/DMARC) */}
            <motion.div
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className="bg-white border border-slate-200 rounded-3xl p-6 shadow-sm flex flex-col h-full"
            >
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-2 bg-blue-50 rounded-xl">
                        <Shield className="w-6 h-6 text-blue-600" />
                    </div>
                    <h3 className="font-bold text-slate-900">Authentication</h3>
                </div>

                <div className="space-y-3 flex-1">
                    {authChecks.map((check) => (
                        <div key={check.id} className={`p-4 rounded-xl border flex items-start gap-3 ${getStatusColor(check.status)}`}>
                            <div className="mt-0.5 flex-shrink-0">{getIcon(check.status)}</div>
                            <div>
                                <h4 className="font-bold text-sm uppercase tracking-wide opacity-80">{check.name}</h4>
                                <p className="text-sm font-medium mt-1">{check.message}</p>
                            </div>
                        </div>
                    ))}
                </div>
            </motion.div>

            {/* 2. Infrastructure & Reputation */}
            <div className="flex flex-col gap-6 h-full">

                {/* MX Check */}
                <motion.div
                    initial={{ opacity: 0, y: 10 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.1 }}
                    className={`flex-1 p-6 rounded-3xl border shadow-sm ${mxCheck ? getStatusColor(mxCheck.status) : 'bg-white border-slate-200'}`}
                >
                    <div className="flex items-center gap-3 mb-2">
                        <Server className="w-6 h-6 opacity-80" />
                        <h3 className="font-bold text-lg">MX Records</h3>
                    </div>
                    <p className="font-medium opacity-90">{mxCheck?.message || "Checking..."}</p>
                </motion.div>

                {/* Blacklist Check */}
                <motion.div
                    initial={{ opacity: 0, y: 10 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 }}
                    className={`flex-1 p-6 rounded-3xl border shadow-sm ${blacklistCheck ? getStatusColor(blacklistCheck.status) : 'bg-white border-slate-200'}`}
                >
                    <div className="flex items-center gap-3 mb-2">
                        <Ban className="w-6 h-6 opacity-80" />
                        <h3 className="font-bold text-lg">Blacklist Status</h3>
                    </div>
                    <p className="font-medium opacity-90">{blacklistCheck?.message || "Checking..."}</p>
                </motion.div>
            </div>

            {/* 3. AI Recommendations */}
            <motion.div
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.3 }}
                className="bg-slate-900 text-white border border-slate-800 rounded-3xl p-6 shadow-xl flex flex-col h-full"
            >
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-2 bg-indigo-500/20 rounded-xl border border-indigo-500/30">
                        <Lightbulb className="w-6 h-6 text-indigo-300" />
                    </div>
                    <h3 className="font-bold text-indigo-100">Optimizations</h3>
                </div>

                <div className="space-y-4 font-medium text-slate-300 text-sm flex-1">
                    {report.recommendations.length > 0 ? (
                        <ul className="space-y-3">
                            {report.recommendations.map((rec, i) => (
                                <li key={i} className="flex gap-3">
                                    <span className="text-indigo-400 font-bold">•</span>
                                    {rec}
                                </li>
                            ))}
                        </ul>
                    ) : (
                        <div className="h-full flex flex-col items-center justify-center text-center opacity-60">
                            <CheckCircle className="w-8 h-8 mb-2" />
                            <p>Everything looks perfect!</p>
                        </div>
                    )}
                </div>
            </motion.div>

        </div>
    );
};
