"use client";

import React, { useState, useRef } from "react";
import Link from "next/link";
import { motion, AnimatePresence, useMotionTemplate, useMotionValue } from "framer-motion";
import {
    Search, Globe, Mail, Shield, Activity, List, Server,
    Wifi, CheckCircle, Lock, LayoutGrid, ArrowRight,
    Fingerprint, Clock, Radio, RotateCw, Link as LinkIcon,
    Eye, Database, FileText, Search as SearchIcon, ShieldCheck
} from "lucide-react";

// Tool Categories
type ToolCategory = 'All' | 'DNS' | 'Email' | 'Security' | 'Network' | 'SEO';

interface Tool {
    id: string;
    name: string;
    desc: string;
    icon: any;
    href: string;
    category: ToolCategory;
    status: 'live' | 'soon';
}

const tools: Tool[] = [
    // DNS Tools
    {
        id: "dns-lookup",
        name: "DNS Lookup",
        desc: "Check A, MX, NS, CNAME records locally & globally.",
        icon: Globe,
        href: "/dns-lookup",
        category: "DNS",
        status: "live"
    },
    {
        id: "whois",
        name: "WHOIS Lookup",
        desc: "Registration data, registrar info, and abuse contacts.",
        icon: Database,
        href: "/whois",
        category: "DNS",
        status: "live"
    },
    {
        id: "reverse-dns",
        name: "Reverse DNS",
        desc: "Perform PTR lookups to find hostnames from IPs.",
        icon: RotateCw,
        href: "/reverse-dns",
        category: "DNS",
        status: "live"
    },
    {
        id: "dns-propagation",
        name: "Propagation",
        desc: "Check DNS records across 25+ global servers.",
        icon: Activity,
        href: "/dns-propagation",
        category: "DNS",
        status: "live"
    },
    {
        id: "dnssec",
        name: "DNSSEC Validator",
        desc: "Verify chain of trust and key signatures.",
        icon: ShieldCheck,
        href: "/dnssec",
        category: "DNS",
        status: "live"
    },
    {
        id: "subdomain-finder",
        name: "Subdomain Finder",
        desc: "Find subdomains using CT logs.",
        icon: Database,
        href: "/subdomain-finder",
        category: "Security",
        status: "live"
    },

    // Email Tools
    {
        id: "email-health",
        name: "Email Health",
        desc: "One-click deliverability audit (SPF, DKIM, DMARC).",
        icon: CheckCircle,
        href: "/email-health",
        category: "Email",
        status: "live"
    },
    {
        id: "mx-lookup",
        name: "MX Lookup",
        desc: "Analyze mail servers and priority configurations.",
        icon: Mail,
        href: "/mx-lookup",
        category: "Email",
        status: "live"
    },
    {
        id: "spf-validator",
        name: "SPF Validator",
        desc: "Check SPF syntax and DNS lookup limits.",
        icon: Fingerprint,
        href: "/spf",
        category: "Email",
        status: "live"
    },
    {
        id: "dmarc-analyzer",
        name: "DMARC Analyzer",
        desc: "Visualize policy enforcement and reporting setup.",
        icon: FileText,
        href: "/dmarc",
        category: "Email",
        status: "live"
    },
    {
        id: "dkim-tester",
        name: "DKIM Tester",
        desc: "Validate DKIM selectors and public keys.",
        icon: Lock,
        href: "/dkim-tester",
        category: "Email",
        status: "live"
    },
    {
        id: "smtp-diag",
        name: "SMTP Diag",
        desc: "Test mail server handshake and TLS support.",
        icon: Server,
        href: "/smtp",
        category: "Email",
        status: "live"
    },

    // Security Tools
    {
        id: "ssl-checker",
        name: "SSL Check",
        desc: "Verify certificate chain, expiry, and validity.",
        icon: Lock,
        href: "/ssl-checker",
        category: "Security",
        status: "live"
    },
    {
        id: "blacklist",
        name: "Blacklist Check",
        desc: "Scan domain/IP against 50+ RBLs (Spamhaus, etc).",
        icon: Shield,
        href: "/blacklist-checker",
        category: "Security",
        status: "live"
    },
    {
        id: "http-headers",
        name: "HTTP Headers",
        desc: "Analyze security headers (CSP, HSTS, X-Frame).",
        icon: List,
        href: "/http-headers",
        category: "Security",
        status: "live"
    },
    {
        id: "port-scanner",
        name: "Port Scanner",
        desc: "Scan common ports (21, 22, 80, 443, 3389) for risks.",
        icon: Activity,
        href: "/port-scanner",
        category: "Security",
        status: "live"
    },

    // Network Tools
    {
        id: "ip-info",
        name: "IP Info",
        desc: "Geolocation, ISP, ASN, and fraud risk data.",
        icon: Globe,
        href: "/ip-info",
        category: "Network",
        status: "live"
    },
    {
        id: "trace",
        name: "MTR / Trace",
        desc: "Trace network path and analyze hop latency.",
        icon: Wifi,
        href: "/trace",
        category: "Network",
        status: "live"
    },
    {
        id: "ping",
        name: "Ping Test",
        desc: "High-precision latency and jitter analysis.",
        icon: Activity,
        href: "/ping",
        category: "Network",
        status: "live"
    },
    {
        id: "cdn-detector",
        name: "CDN Detector",
        desc: "Identify CDN providers and WAFs.",
        icon: Server,
        href: "/cdn-detector",
        category: "Network",
        status: "live"
    },

    // SEO Tools
    {
        id: "redirect-checker",
        name: "Redirect Checker",
        desc: "Follow redirect chains (301, 302) to destination.",
        icon: LinkIcon,
        href: "/redirect-checker",
        category: "SEO",
        status: "live"
    },
    {
        id: "seo-tags",
        name: "Meta Tags",
        desc: "Preview OG tags and Twitter cards for social sharing.",
        icon: Eye,
        href: "/meta-tags",
        category: "SEO",
        status: "live"
    },
];

const CATEGORIES: ToolCategory[] = ['All', 'DNS', 'Email', 'Security', 'Network', 'SEO'];

export const ToolGrid = () => {
    const [selectedCategory, setSelectedCategory] = useState<ToolCategory>('All');
    const [searchQuery, setSearchQuery] = useState("");

    const filteredTools = tools.filter(tool => {
        const matchesCategory = selectedCategory === 'All' || tool.category === selectedCategory;
        const matchesSearch = tool.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
            tool.desc.toLowerCase().includes(searchQuery.toLowerCase());
        return matchesCategory && matchesSearch;
    });

    return (
        <section className="relative py-32 overflow-hidden bg-slate-50" id="tools">
            <div className="absolute inset-0 bg-[url('/grid.svg')] opacity-[0.02]" />
            <div className="absolute inset-0 bg-gradient-to-b from-slate-50 via-white to-slate-50 opacity-80" />

            <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">

                <div className="flex flex-col items-center mb-20 text-center space-y-4">
                    <span className="text-blue-600 font-bold tracking-wider uppercase text-sm bg-blue-50 px-3 py-1 rounded-full border border-blue-100">
                        The Toolkit
                    </span>
                    <h2 className="text-4xl md:text-5xl font-black text-slate-900 tracking-tight font-jakarta">
                        Everything You Need. <br />
                        <span className="text-slate-400">Nothing You Don't.</span>
                    </h2>
                </div>

                {/* Controls Bar */}
                <div className="sticky top-4 z-40 mb-12">
                    <div className="bg-white/70 backdrop-blur-xl border border-white/50 shadow-[0_8px_32px_0_rgba(31,38,135,0.07)] p-2 rounded-3xl flex flex-col lg:flex-row items-center justify-between gap-4">

                        {/* Categories */}
                        <div className="flex items-center gap-1 overflow-x-auto w-full lg:w-auto p-1 scrollbar-hide">
                            {CATEGORIES.map((cat) => (
                                <button
                                    key={cat}
                                    onClick={() => setSelectedCategory(cat)}
                                    className={`relative px-4 py-2.5 rounded-2xl font-bold text-sm whitespace-nowrap transition-all duration-300 z-10
                                        ${selectedCategory === cat ? "text-white shadow-md" : "text-slate-500 hover:text-slate-900 hover:bg-slate-100/50"}
                                    `}
                                >
                                    {selectedCategory === cat && (
                                        <motion.div
                                            layoutId="activeTab"
                                            className="absolute inset-0 bg-blue-600 rounded-2xl -z-10"
                                            transition={{ type: "spring", bounce: 0.2, duration: 0.6 }}
                                        />
                                    )}
                                    {cat}
                                </button>
                            ))}
                        </div>

                        {/* Search */}
                        <div className="relative w-full lg:w-72">
                            <SearchIcon className="absolute left-4 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-400" />
                            <input
                                type="text"
                                placeholder="Search generic tools..."
                                value={searchQuery}
                                onChange={(e) => setSearchQuery(e.target.value)}
                                suppressHydrationWarning
                                className="w-full h-11 pl-10 pr-4 bg-slate-100/50 border border-transparent focus:bg-white focus:border-blue-200 focus:ring-4 focus:ring-blue-500/10 rounded-2xl text-sm font-semibold outline-none transition-all placeholder:text-slate-400"
                            />
                        </div>
                    </div>
                </div>

                {/* Grid */}
                <motion.div
                    layout
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
                >
                    <AnimatePresence mode="popLayout">
                        {filteredTools.map((tool) => (
                            <SpotlightCard key={tool.id} tool={tool} />
                        ))}
                    </AnimatePresence>
                </motion.div>

                {filteredTools.length === 0 && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="text-center py-20"
                    >
                        <div className="w-16 h-16 bg-slate-100 rounded-full flex items-center justify-center mx-auto mb-4 text-slate-400">
                            <SearchIcon className="w-8 h-8" />
                        </div>
                        <h3 className="text-slate-900 font-bold text-lg">No tools found</h3>
                        <p className="text-slate-500">Try adjusting your search criteria.</p>
                    </motion.div>
                )}
            </div>
        </section>
    );
};

// --- Spotlight Card Component ---

function SpotlightCard({ tool }: { tool: Tool }) {
    const mouseX = useMotionValue(0);
    const mouseY = useMotionValue(0);

    function handleMouseMove({ currentTarget, clientX, clientY }: React.MouseEvent) {
        const { left, top } = currentTarget.getBoundingClientRect();
        mouseX.set(clientX - left);
        mouseY.set(clientY - top);
    }

    return (
        <motion.div
            layout
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            exit={{ opacity: 0, scale: 0.9 }}
            transition={{ duration: 0.3 }}
        >
            <Link
                href={tool.status === 'live' ? tool.href : '#'}
                className={`group relative block h-full rounded-3xl bg-white border border-slate-200 overflow-hidden ${tool.status === 'live' ? 'cursor-pointer' : 'cursor-not-allowed opacity-60'
                    }`}
                onMouseMove={handleMouseMove}
            >
                {/* Spotlight Gradient */}
                {tool.status === 'live' && (
                    <motion.div
                        className="pointer-events-none absolute -inset-px opacity-0 transition duration-300 group-hover:opacity-100"
                        style={{
                            background: useMotionTemplate`
                                radial-gradient(
                                    600px circle at ${mouseX}px ${mouseY}px,
                                    rgba(37, 99, 235, 0.1),
                                    transparent 40%
                                )
                            `,
                        }}
                    />
                )}

                <div className="relative h-full p-6 flex flex-col z-10">
                    <div className="flex items-start justify-between mb-4">
                        <div className={`
                            p-3 rounded-2xl bg-slate-50 text-slate-600 
                            group-hover:bg-blue-600 group-hover:text-white group-hover:scale-110 group-hover:shadow-lg group-hover:shadow-blue-500/20
                            transition-all duration-300
                        `}>
                            <tool.icon className="w-6 h-6" />
                        </div>
                        {tool.status === 'soon' && (
                            <span className="text-[10px] uppercase font-black tracking-widest bg-slate-100 text-slate-500 px-2 py-1 rounded-lg">Soon</span>
                        )}
                    </div>

                    <h3 className="text-lg font-bold text-slate-900 mb-2 font-jakarta">
                        {tool.name}
                    </h3>
                    <p className="text-sm text-slate-500 font-medium leading-relaxed mb-6 flex-1">
                        {tool.desc}
                    </p>

                    {tool.status === 'live' && (
                        <div className="flex items-center gap-2 text-xs font-bold text-blue-600 uppercase tracking-wide opacity-0 -translate-x-2 group-hover:opacity-100 group-hover:translate-x-0 transition-all duration-300">
                            Launch Tool <ArrowRight className="w-3.5 h-3.5" />
                        </div>
                    )}
                </div>
            </Link>
        </motion.div>
    );
}
