"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Shield, ShieldAlert, Zap, Server, Copy, Check, FileCode, AlertTriangle } from "lucide-react";

export interface HeaderResult {
    key: string;
    value: string;
    category: 'security' | 'caching' | 'server' | 'other';
}

interface HeaderDashboardProps {
    headers: HeaderResult[];
    rawHeaders: string;
}

export const HeaderDashboard: React.FC<HeaderDashboardProps> = ({ headers, rawHeaders }) => {
    const [showRaw, setShowRaw] = useState(false);
    const [copied, setCopied] = useState<string | null>(null);

    const handleCopy = (text: string, id: string) => {
        navigator.clipboard.writeText(text);
        setCopied(id);
        setTimeout(() => setCopied(null), 2000);
    };

    // Categorize
    const securityHeaders = headers.filter(h => h.category === 'security');
    const cacheHeaders = headers.filter(h => h.category === 'caching');
    const serverHeaders = headers.filter(h => h.category === 'server');
    const otherHeaders = headers.filter(h => h.category === 'other');

    // Missing Critical Security Headers Check
    const criticalSecurity = [
        'Content-Security-Policy',
        'Strict-Transport-Security',
        'X-Frame-Options',
        'X-Content-Type-Options',
        'Referrer-Policy'
    ];

    // Convert headers to lowercase keys for comparison
    const existingKeys = new Set(headers.map(h => h.key.toLowerCase()));
    const missingSecurity = criticalSecurity.filter(k => !existingKeys.has(k.toLowerCase()));

    return (
        <div className="space-y-6">

            {/* Control Bar */}
            <div className="flex items-center justify-end">
                <button
                    onClick={() => setShowRaw(!showRaw)}
                    className={`flex items-center gap-2 px-4 py-2 rounded-lg text-sm font-bold border transition-colors
                        ${showRaw ? "bg-slate-900 text-white border-slate-900" : "bg-white text-slate-600 border-slate-200 hover:bg-slate-50"}
                    `}
                >
                    <FileCode className="w-4 h-4" />
                    {showRaw ? "View Structured" : "View Raw Response"}
                </button>
            </div>

            {showRaw ? (
                <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="bg-slate-900 rounded-xl p-6 overflow-x-auto shadow-inner">
                    <pre className="text-sm font-mono text-blue-300 whitespace-pre-wrap break-all leading-relaxed">
                        {rawHeaders}
                    </pre>
                </motion.div>
            ) : (
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 items-start">

                    {/* 1. Security Headers */}
                    <div className="space-y-6">
                        <SectionTitle icon={<Shield className="w-5 h-5 text-emerald-500" />} title="Security Policy" />
                        <div className="space-y-3">
                            {securityHeaders.length === 0 && <EmptyState text="No security headers found." />}
                            {securityHeaders.map((h, i) => (
                                <HeaderCard key={i} header={h} onCopy={handleCopy} copied={copied} />
                            ))}

                            {/* Missing Warnings */}
                            {missingSecurity.length > 0 && (
                                <div className="bg-amber-50 border border-amber-100 rounded-xl p-4 mt-4">
                                    <div className="flex items-center gap-2 text-xs font-bold text-amber-600 uppercase mb-3">
                                        <AlertTriangle className="w-4 h-4" /> Missing Recommendations
                                    </div>
                                    <div className="space-y-2">
                                        {missingSecurity.map(k => (
                                            <div key={k} className="flex items-center gap-2 text-sm text-slate-600">
                                                <div className="w-1.5 h-1.5 rounded-full bg-amber-400" />
                                                {k}
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* 2. Cache & Performance */}
                    <div className="space-y-6">
                        <SectionTitle icon={<Zap className="w-5 h-5 text-blue-500" />} title="Cache & Performance" />
                        <div className="space-y-3">
                            {cacheHeaders.length === 0 && <EmptyState text="No caching headers found." />}
                            {cacheHeaders.map((h, i) => (
                                <HeaderCard key={i} header={h} onCopy={handleCopy} copied={copied} />
                            ))}
                        </div>
                    </div>

                    {/* 3. Server & Connection */}
                    <div className="space-y-6">
                        <SectionTitle icon={<Server className="w-5 h-5 text-purple-500" />} title="Server & Connection" />
                        <div className="space-y-3">
                            {serverHeaders.length === 0 && otherHeaders.length === 0 && <EmptyState text="No server headers found." />}
                            {serverHeaders.map((h, i) => (
                                <HeaderCard key={i} header={h} onCopy={handleCopy} copied={copied} />
                            ))}
                            {otherHeaders.map((h, i) => (
                                <HeaderCard key={i} header={h} onCopy={handleCopy} copied={copied} />
                            ))}
                        </div>
                    </div>

                </div>
            )}
        </div>
    );
};

const SectionTitle = ({ icon, title }: { icon: React.ReactNode, title: string }) => (
    <div className="flex items-center gap-2 pb-2 border-b border-slate-100">
        <div className="p-2 bg-white rounded-lg border border-slate-100 shadow-sm">
            {icon}
        </div>
        <h3 className="font-bold text-slate-800">{title}</h3>
    </div>
);

const HeaderCard = ({ header, onCopy, copied }: { header: HeaderResult, onCopy: (t: string, id: string) => void, copied: string | null }) => (
    <div className="bg-white border border-slate-200 rounded-xl p-3 hover:border-blue-300 transition-colors group relative overflow-hidden">
        <div className="flex justify-between items-start gap-4">
            <div className="w-full min-w-0">
                <div className="text-xs font-bold text-slate-500 uppercase tracking-wider mb-1 mr-6 truncate" title={header.key}>
                    {header.key}
                </div>
                <div className="font-mono text-sm text-blue-600 break-all leading-tight">
                    {header.value}
                </div>
            </div>
            <button
                onClick={() => onCopy(header.value, header.key)}
                className="absolute top-2 right-2 p-1.5 text-slate-300 hover:text-blue-500 hover:bg-blue-50 rounded-lg transition-all opacity-0 group-hover:opacity-100"
            >
                {copied === header.key ? <Check className="w-3.5 h-3.5" /> : <Copy className="w-3.5 h-3.5" />}
            </button>
        </div>
    </div>
);

const EmptyState = ({ text }: { text: string }) => (
    <div className="p-4 rounded-xl border border-dashed border-slate-200 text-slate-400 text-sm italic text-center">
        {text}
    </div>
);
