"use client";

import React, { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Loader2, Search, Copy, Check, Globe, ShieldCheck, Zap } from "lucide-react";

import { IPIntelligenceGrid, IPInfoData } from "@/components/ip-info/IPIntelligenceGrid";

export default function IPInfoClient() {
    const [ipInput, setIpInput] = useState("");
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState<IPInfoData | null>(null);
    const [error, setError] = useState("");
    const [copied, setCopied] = useState(false);

    // Auto-detect on load
    useEffect(() => {
        fetchInfo();
    }, []);

    const fetchInfo = async (ip?: string) => {
        setLoading(true);
        setError("");

        try {
            const url = ip ? `/api/ip-info?ip=${ip}` : '/api/ip-info';
            const res = await fetch(url);
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to fetch IP data");
            }
            setData(json);
            // If checking "my ip", clear input for cleaner look
            if (!ip) setIpInput("");
        } catch (err: any) {
            setError(err.message);
            setData(null);
        } finally {
            setLoading(false);
        }
    };

    const handleSearch = () => {
        if (ipInput) fetchInfo(ipInput);
    };

    const handleMyIp = () => {
        setIpInput("");
        fetchInfo();
    };

    const handleCopy = () => {
        if (data?.ip) {
            navigator.clipboard.writeText(data.ip);
            setCopied(true);
            setTimeout(() => setCopied(false), 2000);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header Section with Aurora Gradient */}
            <div className="relative pt-24 pb-32 overflow-hidden bg-white">
                {/* Background Effects */}
                <div className="absolute inset-0 z-0">
                    <div className="absolute top-0 inset-x-0 h-px bg-gradient-to-r from-transparent via-slate-200 to-transparent" />
                    <div className="absolute -top-[30%] -left-[10%] w-[70%] h-[70%] rounded-full bg-blue-100/30 blur-[100px]" />
                    <div className="absolute top-[10%] -right-[10%] w-[50%] h-[60%] rounded-full bg-purple-100/30 blur-[100px]" />
                    <div className="absolute inset-0 opacity-[0.03]" style={{ backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23000000' fill-opacity='1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2v-4h4v-2h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")` }} />
                </div>

                <div className="max-w-4xl mx-auto px-4 text-center relative z-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.5 }}>
                        <div className="inline-flex items-center justify-center p-3 mb-6 bg-white border border-slate-100 shadow-xl shadow-blue-500/5 rounded-2xl text-blue-600">
                            <Globe className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-6xl font-extrabold text-slate-900 mb-6 tracking-tight font-jakarta">
                            IP Intelligence <span className="text-transparent bg-clip-text bg-gradient-to-r from-blue-600 to-purple-600">& Geo-Data</span>
                        </h1>
                        <p className="text-lg text-slate-500 max-w-2xl mx-auto mb-10 leading-relaxed">
                            Complete diagnostic precision for any IPv4 address.
                            Instantly analyze ASN, location, and security risk factors.
                        </p>

                        {/* Search Interface */}
                        <div className="flex flex-col sm:flex-row items-stretch gap-3 max-w-2xl mx-auto bg-white/70 backdrop-blur-xl p-2 rounded-2xl shadow-2xl shadow-blue-900/5 border border-white/50 ring-1 ring-slate-200/50">
                            <div className="flex-1 relative group">
                                <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                                <input
                                    type="text"
                                    placeholder="Enter any IP address..."
                                    value={ipInput}
                                    onChange={(e) => setIpInput(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="w-full h-12 pl-12 pr-4 bg-transparent text-lg font-medium text-slate-900 placeholder:text-slate-400 outline-none rounded-xl hover:bg-slate-50/50 focus:bg-slate-50 transition-colors"
                                    suppressHydrationWarning
                                />
                            </div>

                            <div className="flex items-center gap-2 shrink-0">
                                <button
                                    onClick={handleSearch}
                                    disabled={loading}
                                    className="h-12 px-6 bg-slate-900 hover:bg-slate-800 text-white font-bold rounded-xl transition-all shadow-lg shadow-slate-900/20 active:scale-95 disabled:opacity-70 disabled:active:scale-100 flex items-center gap-2"
                                >
                                    {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : "Analyze"}
                                </button>

                                <button
                                    onClick={handleMyIp}
                                    className="h-12 px-4 bg-blue-50 hover:bg-blue-100 text-blue-600 font-bold rounded-xl transition-colors border border-blue-200/50 flex items-center gap-2 whitespace-nowrap"
                                    title="Detect My IP"
                                >
                                    <Zap className="w-4 h-4 fill-current" />
                                    <span className="hidden sm:inline">My IP</span>
                                </button>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </div>

            {/* Main Content Area - Overlaps Header */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 -mt-20 relative z-20 pb-20 w-full">
                {error && (
                    <motion.div
                        initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}
                        className="max-w-xl mx-auto bg-red-50/90 backdrop-blur border border-red-100/50 text-red-600 p-4 rounded-xl text-center shadow-lg mb-8 font-medium"
                    >
                        {error}
                    </motion.div>
                )}

                <AnimatePresence mode="wait">
                    {data && !loading && (
                        <motion.div
                            key="results"
                            initial={{ opacity: 0, y: 20 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ duration: 0.4 }}
                            className="space-y-8"
                        >
                            {/* Hero IP Card */}
                            <div className="bg-white/80 backdrop-blur-md rounded-3xl border border-white/50 shadow-[0_8px_30px_rgb(0,0,0,0.04)] p-8 md:p-12 text-center relative overflow-hidden group">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 via-purple-500 to-blue-500 opacity-0 group-hover:opacity-100 transition-opacity" />

                                <div className="inline-flex items-center gap-2 text-sm font-bold text-slate-400 uppercase tracking-wider mb-4">
                                    Target Identifier
                                </div>
                                <div className="flex flex-col items-center justify-center gap-4">
                                    <h2 className="text-5xl md:text-7xl font-black text-slate-900 tracking-tighter drop-shadow-sm font-jakarta">
                                        {data.ip}
                                    </h2>

                                    <div className="flex items-center gap-3">
                                        <button
                                            onClick={handleCopy}
                                            className="flex items-center gap-2 px-4 py-2 rounded-full bg-slate-50 hover:bg-slate-100 text-slate-500 hover:text-blue-600 transition-all font-medium text-sm border border-slate-100"
                                        >
                                            {copied ? <Check className="w-4 h-4 text-green-500" /> : <Copy className="w-4 h-4" />}
                                            {copied ? "Copied to Clipboard" : "Copy Address"}
                                        </button>

                                        {data.ip === '127.0.0.1' || data.ip.startsWith('192.168') ? (
                                            <span className="flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-orange-50 text-orange-600 text-xs font-bold border border-orange-100">
                                                <ShieldCheck className="w-3 h-3" /> Local Network
                                            </span>
                                        ) : (
                                            <span className="flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-green-50 text-green-600 text-xs font-bold border border-green-100">
                                                <Globe className="w-3 h-3" /> Public Address
                                            </span>
                                        )}
                                    </div>
                                </div>
                            </div>

                            {/* Bento Grid */}
                            <IPIntelligenceGrid data={data} />
                        </motion.div>
                    )}
                </AnimatePresence>
            </main>


        </div>
    );
}
