"use client";

import React from "react";
import Link from "next/link";
import { Github, Twitter, Bot, Server, ArrowUpRight, MessageCircle } from "lucide-react";
import { motion } from "framer-motion";
import { SystemStatus } from "./SystemStatus";

export const Footer = () => {
    return (
        <footer className="bg-[#020617] relative overflow-hidden pt-32 pb-12 border-t border-slate-900">
            {/* Animated Grid Background */}
            <div className="absolute inset-0 opacity-10 pointer-events-none">
                <div
                    className="absolute inset-0"
                    style={{
                        backgroundImage: 'linear-gradient(to right, #334155 1px, transparent 1px), linear-gradient(to bottom, #334155 1px, transparent 1px)',
                        backgroundSize: '40px 40px',
                        transform: 'perspective(500px) rotateX(60deg) translateY(-100px) scale(3)',
                        transformOrigin: 'top center',
                        animation: 'gridMove 20s linear infinite'
                    }}
                />
            </div>
            <style jsx>{`
                @keyframes gridMove {
                    0% { transform: perspective(500px) rotateX(60deg) translateY(0) scale(3); }
                    100% { transform: perspective(500px) rotateX(60deg) translateY(40px) scale(3); }
                }
            `}</style>

            {/* Gradient Glows */}
            <div className="absolute top-0 left-1/2 -translate-x-1/2 w-[1000px] h-[400px] bg-blue-600/10 blur-[120px] rounded-full pointer-events-none" />

            <div className="max-w-[1400px] mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
                <div className="grid grid-cols-1 lg:grid-cols-12 gap-12 mb-24">

                    {/* Brand & System Status */}
                    <div className="lg:col-span-3 space-y-8">
                        <div>
                            <Link href="/" className="inline-block group">
                                <div className="flex items-center gap-3 text-2xl font-black text-white tracking-tight">
                                    <div className="relative">
                                        <div className="absolute inset-0 bg-blue-500 blur-lg opacity-20 group-hover:opacity-40 transition-opacity" />
                                        <span className="relative bg-gradient-to-br from-blue-600 to-indigo-600 w-12 h-12 rounded-xl flex items-center justify-center border border-white/10 shadow-2xl group-hover:scale-105 transition-transform duration-300">
                                            <Server className="w-6 h-6 text-white" />
                                        </span>
                                    </div>
                                    <span className="group-hover:text-blue-400 transition-colors duration-300">DNS Tool</span>
                                </div>
                            </Link>
                            <p className="mt-6 text-slate-400 text-sm leading-relaxed max-w-sm">
                                Advanced network intelligence for the modern web. Real-time diagnostics with enterprise-grade precision.
                            </p>
                        </div>

                        {/* The Dashboard Widget */}
                        <motion.div
                            initial={{ opacity: 0, y: 20 }}
                            whileInView={{ opacity: 1, y: 0 }}
                            viewport={{ once: true }}
                            transition={{ duration: 0.8, delay: 0.2 }}
                        >
                            <SystemStatus />
                        </motion.div>
                    </div>

                    {/* Links Columns */}
                    <div className="lg:col-span-9 grid grid-cols-2 md:grid-cols-3 xl:grid-cols-5 gap-8 pt-4">
                        <FooterColumn title="DNS" links={[
                            { name: "DNS Lookup", href: "/dns-lookup" },
                            { name: "Propagation", href: "/dns-propagation" },
                            { name: "Reverse DNS", href: "/reverse-dns" },
                            { name: "DNSSEC", href: "/dnssec" },
                            { name: "TXT Record", href: "/txt-record" },
                            { name: "MX Lookup", href: "/mx-lookup" },
                        ]} />

                        <FooterColumn title="Email" links={[
                            { name: "WHOIS", href: "/whois" },
                            { name: "DMARC", href: "/dmarc" },
                            { name: "SPF", href: "/spf" },
                            { name: "DKIM Tester", href: "/dkim-tester" },
                            { name: "SMTP Test", href: "/smtp" },
                            { name: "Email Health", href: "/email-health" },
                        ]} />

                        <FooterColumn title="Security" links={[
                            { name: "SSL Checker", href: "/ssl-checker" },
                            { name: "Blacklist", href: "/blacklist-checker" },
                            { name: "Subdomain", href: "/subdomain-finder" },
                            { name: "Domain Expiry", href: "/domain-expiry" },
                        ]} />

                        <FooterColumn title="Network" links={[
                            { name: "Port Scanner", href: "/port-scanner" },
                            { name: "IP Info", href: "/ip-info" },
                            { name: "Ping", href: "/ping" },
                            { name: "Trace", href: "/trace" },
                            { name: "CDN Detector", href: "/cdn-detector" },
                        ]} />

                        <FooterColumn title="SEO" links={[
                            { name: "Meta Tags", href: "/meta-tags" },
                            { name: "Redirects", href: "/redirect-checker" },
                            { name: "Headers", href: "/http-headers" },
                        ]} />
                    </div>
                </div>

                {/* Bottom Bar */}
                <div className="pt-8 border-t border-slate-900 flex flex-col md:flex-row justify-between items-center gap-6">
                    <div className="flex flex-wrap justify-center md:justify-start gap-6 text-sm text-slate-500">
                        <Link href="/" className="hover:text-blue-400 transition-colors">Home</Link>
                        <Link href="/about" className="hover:text-blue-400 transition-colors">About</Link>
                        <Link href="/contact" className="hover:text-blue-400 transition-colors">Contact</Link>
                        <span>© {new Date().getFullYear()} DNS Tool Inc.</span>
                    </div>

                    <div className="flex items-center gap-4">
                        <SocialLink href="#" icon={<Github className="w-5 h-5" />} label="GitHub" />
                        <SocialLink href="#" icon={<Twitter className="w-5 h-5" />} label="Twitter" />
                        <SocialLink href="#" icon={<MessageCircle className="w-5 h-5" />} label="Discord" />
                    </div>
                </div>
            </div>
        </footer>
    );
};

const FooterColumn = ({ title, links }: { title: string, links: { name: string, href: string }[] }) => (
    <div>
        <h4 className="font-bold text-white mb-6 flex items-center gap-2">
            <span className="w-1 h-4 bg-blue-600 rounded-full" />
            {title}
        </h4>
        <ul className="space-y-3">
            {links.map((link) => (
                <li key={link.name}>
                    <Link href={link.href} className="text-slate-400 hover:text-white transition-colors flex items-center group w-fit text-sm">
                        <span className="relative overflow-hidden">
                            <span className="inline-block transition-transform duration-300 group-hover:-translate-y-full">{link.name}</span>
                            <span className="absolute top-0 left-0 inline-block transition-transform duration-300 translate-y-full group-hover:translate-y-0 text-blue-400 font-medium">
                                {link.name}
                            </span>
                        </span>
                    </Link>
                </li>
            ))}
        </ul>
    </div>
);

const SocialLink = ({ href, icon, label }: { href: string; icon: React.ReactNode, label: string }) => {
    return (
        <motion.a
            href={href}
            whileHover={{ scale: 1.1, rotate: 5 }}
            whileTap={{ scale: 0.9 }}
            className="w-12 h-12 rounded-xl bg-slate-900 border border-slate-800 text-slate-400 flex items-center justify-center hover:bg-blue-600 hover:text-white hover:border-blue-500 transition-all shadow-lg group relative"
            aria-label={label}
        >
            {icon}
            {/* Tooltip */}
            <span className="absolute -top-10 left-1/2 -translate-x-1/2 px-2 py-1 bg-slate-800 text-white text-xs rounded opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none whitespace-nowrap border border-slate-700">
                {label}
            </span>
        </motion.a>
    );
};
