"use client";

import React, { useState, useEffect } from "react";
import Link from "next/link";
import { motion, AnimatePresence } from "framer-motion";
import { Menu, X, ChevronDown, Server, Zap, Globe, ChevronRight } from "lucide-react";

export const Navbar = () => {
    const [scrolled, setScrolled] = useState(false);
    const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
    const [hoveredCategory, setHoveredCategory] = useState<string | null>(null);
    const [mobileExpandedCategory, setMobileExpandedCategory] = useState<string | null>(null);

    useEffect(() => {
        const handleScroll = () => setScrolled(window.scrollY > 20);
        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, []);

    const toggleMobileCategory = (category: string) => {
        setMobileExpandedCategory(prev => prev === category ? null : category);
    };

    const navItems = {
        DNS: [
            { name: "DNS Lookup", href: "/dns-lookup", desc: "Check DNS records instantly" },
            { name: "Propagation", href: "/dns-propagation", desc: "Global DNS status check" },
            { name: "Reverse DNS", href: "/reverse-dns", desc: "Convert IP to Hostname" },
            { name: "DNSSEC", href: "/dnssec", desc: "Verify domain security" },
            { name: "TXT Record", href: "/txt-record", desc: "View TXT records" },
            { name: "MX Lookup", href: "/mx-lookup", desc: "Find mail servers" },
        ],
        Email: [
            { name: "WHOIS", href: "/whois", desc: "Domain registration info" },
            { name: "DMARC", href: "/dmarc", desc: "Email auth verification" },
            { name: "SPF", href: "/spf", desc: "Sender policy check" },
            { name: "DKIM Tester", href: "/dkim-tester", desc: "Key verification" },
            { name: "SMTP Test", href: "/smtp", desc: "Server diagnostics" },
            { name: "Email Health", href: "/email-health", desc: "Overall domain status" },
        ],
        Security: [
            { name: "SSL Checker", href: "/ssl-checker", desc: "Verify SSL certificates" },
            { name: "Blacklist", href: "/blacklist-checker", desc: "IP reputation scan" },
            { name: "Subdomain", href: "/subdomain-finder", desc: "Discover subdomains" },
            { name: "Domain Expiry", href: "/domain-expiry", desc: "Check expiration dates" },
        ],
        Network: [
            { name: "Port Scanner", href: "/port-scanner", desc: "Check open ports" },
            { name: "IP Info", href: "/ip-info", desc: "Geolocation & ISP" },
            { name: "Ping", href: "/ping", desc: "Measure latency" },
            { name: "Trace", href: "/trace", desc: "Hop-by-hop analysis" },
            { name: "CDN Detector", href: "/cdn-detector", desc: "Identify CDN provider" },
        ],
        SEO: [
            { name: "Meta Tags", href: "/meta-tags", desc: "Analyze page metadata" },
            { name: "Redirects", href: "/redirect-checker", desc: "Trace URL path" },
            { name: "Headers", href: "/http-headers", desc: "Inspect HTTP response" },
        ]
    };

    return (
        <>
            <motion.nav
                initial={{ y: -100 }}
                animate={{ y: 0 }}
                className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${scrolled ? "py-4" : "py-6"}`}
            >
                <div className="max-w-[1400px] mx-auto px-4">
                    <div
                        className={`relative rounded-full transition-all duration-300 flex items-center justify-between px-6 py-3 ${scrolled
                                ? "bg-white/90 backdrop-blur-xl shadow-lg border border-white/50"
                                : "bg-transparent"
                            }`}
                    >
                        {/* Logo */}
                        <Link href="/" className="flex items-center gap-2 group mr-8">
                            <div className="w-10 h-10 bg-slate-900 rounded-xl flex items-center justify-center text-white shadow-lg">
                                <Server className="w-5 h-5" />
                            </div>
                            <span className="text-xl font-bold text-slate-900 hidden sm:block">DNS Tool</span>
                        </Link>

                        {/* Desktop Nav */}
                        <div className="hidden xl:flex items-center gap-1 bg-slate-100/50 p-1 rounded-full border border-slate-200/50 backdrop-blur-sm">
                            <Link href="/" className="px-5 py-2 text-sm font-medium text-slate-600 hover:text-blue-600 rounded-full hover:bg-white transition-all">
                                Home
                            </Link>

                            {Object.entries(navItems).map(([category, items]) => (
                                <div
                                    key={category}
                                    className="relative group"
                                    onMouseEnter={() => setHoveredCategory(category)}
                                    onMouseLeave={() => setHoveredCategory(null)}
                                >
                                    <button className="flex items-center gap-1 px-5 py-2 text-sm font-medium text-slate-600 hover:text-blue-600 rounded-full hover:bg-white transition-all cursor-default">
                                        {category}
                                        <ChevronDown className="w-3 h-3 opacity-50 group-hover:rotate-180 transition-transform" />
                                    </button>

                                    {/* Dropdown */}
                                    <div className="absolute top-full left-1/2 -translate-x-1/2 pt-4 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 w-[300px]">
                                        <div className="bg-white rounded-2xl shadow-xl ring-1 ring-slate-900/5 p-2 overflow-hidden">
                                            <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-indigo-500" />
                                            {items.map((item) => (
                                                <Link
                                                    key={item.name}
                                                    href={item.href}
                                                    className="flex items-start gap-3 p-3 rounded-xl hover:bg-slate-50 transition-colors group/item"
                                                >
                                                    <div className="mt-1 w-full">
                                                        <div className="text-sm font-medium text-slate-900 group-hover/item:text-blue-600 flex justify-between items-center w-full">
                                                            {item.name}
                                                        </div>
                                                        <div className="text-xs text-slate-500 line-clamp-1">{item.desc}</div>
                                                    </div>
                                                </Link>
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            ))}

                            <Link href="/about" className="px-5 py-2 text-sm font-medium text-slate-600 hover:text-blue-600 rounded-full hover:bg-white transition-all">
                                About
                            </Link>
                            <Link href="/contact" className="px-5 py-2 text-sm font-medium text-slate-600 hover:text-blue-600 rounded-full hover:bg-white transition-all">
                                Contact
                            </Link>
                        </div>

                        {/* Mobile Toggle & CTA */}
                        <div className="flex items-center gap-4">
                            <Link href="#" className="hidden sm:flex items-center gap-2 bg-slate-900 text-white px-5 py-2.5 rounded-full text-sm font-bold hover:bg-slate-800 transition-all shadow-lg hover:shadow-xl hover:scale-105 active:scale-95">
                                <Globe className="w-4 h-4 text-blue-400" />
                                Main Web
                            </Link>
                            <button
                                onClick={() => setMobileMenuOpen(true)}
                                className="xl:hidden p-2 text-slate-600 bg-white/50 rounded-full hover:bg-white transition-all"
                            >
                                <Menu className="w-6 h-6" />
                            </button>
                        </div>
                    </div>
                </div>
            </motion.nav>

            {/* Mobile Drawer */}
            <AnimatePresence>
                {mobileMenuOpen && (
                    <>
                        <motion.div
                            initial={{ opacity: 0 }}
                            animate={{ opacity: 1 }}
                            exit={{ opacity: 0 }}
                            onClick={() => setMobileMenuOpen(false)}
                            className="fixed inset-0 bg-slate-900/60 backdrop-blur-sm z-[60]"
                        />
                        <motion.div
                            initial={{ x: "-100%" }}
                            animate={{ x: 0 }}
                            exit={{ x: "-100%" }}
                            transition={{ type: "spring", damping: 30, stiffness: 300 }}
                            className="fixed top-0 left-0 bottom-0 w-[320px] bg-white z-[70] shadow-2xl flex flex-col overflow-hidden"
                        >
                            <div className="p-6 flex items-center justify-between border-b border-slate-100 bg-slate-50/50">
                                <div className="flex items-center gap-2">
                                    <div className="w-8 h-8 bg-slate-900 rounded-lg flex items-center justify-center text-white">
                                        <Server className="w-4 h-4" />
                                    </div>
                                    <span className="font-bold text-lg text-slate-900">Menu</span>
                                </div>
                                <button
                                    onClick={() => setMobileMenuOpen(false)}
                                    className="p-2 text-slate-400 hover:text-slate-900 hover:bg-slate-200 rounded-full transition-colors"
                                >
                                    <X className="w-5 h-5" />
                                </button>
                            </div>

                            <div className="flex-1 overflow-y-auto p-4 space-y-1">
                                <MobileLink href="/" onClick={() => setMobileMenuOpen(false)}>Home</MobileLink>

                                {Object.entries(navItems).map(([category, items]) => {
                                    const isOpen = mobileExpandedCategory === category;
                                    return (
                                        <div key={category} className="overflow-hidden rounded-xl">
                                            <button
                                                onClick={() => toggleMobileCategory(category)}
                                                className={`w-full flex items-center justify-between px-4 py-3 font-medium transition-colors ${isOpen ? 'bg-blue-50 text-blue-700' : 'text-slate-600 hover:bg-slate-50'
                                                    }`}
                                            >
                                                <span className="flex items-center gap-2">
                                                    {category}
                                                </span>
                                                <ChevronRight className={`w-4 h-4 transition-transform duration-200 ${isOpen ? 'rotate-90 text-blue-500' : 'text-slate-400'}`} />
                                            </button>

                                            <AnimatePresence>
                                                {isOpen && (
                                                    <motion.div
                                                        initial={{ height: 0, opacity: 0 }}
                                                        animate={{ height: "auto", opacity: 1 }}
                                                        exit={{ height: 0, opacity: 0 }}
                                                        transition={{ duration: 0.2 }}
                                                        className="overflow-hidden bg-slate-50"
                                                    >
                                                        <div className="px-4 py-2 space-y-1">
                                                            {items.map((item) => (
                                                                <Link
                                                                    key={item.name}
                                                                    href={item.href}
                                                                    onClick={() => setMobileMenuOpen(false)}
                                                                    className="block py-2 pl-4 text-sm text-slate-500 hover:text-blue-600 transition-colors border-l-2 border-slate-200 hover:border-blue-500"
                                                                >
                                                                    {item.name}
                                                                </Link>
                                                            ))}
                                                        </div>
                                                    </motion.div>
                                                )}
                                            </AnimatePresence>
                                        </div>
                                    );
                                })}

                                <div className="h-px bg-slate-100 my-4" />
                                <MobileLink href="/about" onClick={() => setMobileMenuOpen(false)}>About</MobileLink>
                                <MobileLink href="/contact" onClick={() => setMobileMenuOpen(false)}>Contact</MobileLink>
                                <div className="h-4" />
                                <Link
                                    href="#"
                                    className="block px-4 py-3 rounded-xl bg-slate-900 text-white font-bold text-center shadow-lg transform active:scale-95 transition-all flex items-center justify-center gap-2"
                                    onClick={() => setMobileMenuOpen(false)}
                                >
                                    <Globe className="w-4 h-4 text-blue-400" />
                                    Main Web
                                </Link>
                            </div>
                        </motion.div>
                    </>
                )}
            </AnimatePresence>
        </>
    );
};

const MobileLink = ({ href, children, onClick, className = "" }: any) => (
    <Link
        href={href}
        onClick={onClick}
        className={`block px-4 py-3 rounded-xl hover:bg-slate-50 font-medium text-slate-600 hover:text-blue-600 transition-colors ${className}`}
    >
        {children}
    </Link>
);
