"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Eye, Loader2, Sparkles, LayoutTemplate, AlertCircle } from "lucide-react";

import { SocialPreview, MetaData } from "@/components/meta-tags/SocialPreview";
import { SEOScorecard } from "@/components/meta-tags/SEOScorecard";

// Extend MetaData locally
interface ExtendedMetaData extends MetaData {
    analysis: {
        score: number;
        h1Count: number;
        suggestions: { severity: 'critical' | 'warning' | 'info'; message: string; fix: string }[];
    }
}

export default function MetaTagsClient() {
    const [url, setUrl] = useState("");
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState<ExtendedMetaData | null>(null);
    const [error, setError] = useState<string | null>(null);

    const handleInspect = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!url) return;

        setLoading(true);
        setError(null);
        setData(null);

        try {
            const response = await fetch('/api/scrape/meta', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ url: url.trim() }),
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || "Inspection failed");
            }

            setData(result);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <LayoutTemplate className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                SEO & Social <span className="text-blue-600">Meta Inspector</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Preview how your website appears on search engines and social media platforms. Validate Title, Description, and OG Tags.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleInspect} className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Eye className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Website URL (e.g., airbnb.com)"
                                value={url}
                                onChange={(e) => setUrl(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !url}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Sparkles className="w-5 h-5 fill-current" />}
                            {loading ? "Inspecting..." : "Inspect Meta"}
                        </button>
                    </form>
                </div>

                {loading && (
                    <div className="flex flex-col items-center justify-center py-20 text-center space-y-4">
                        <Loader2 className="w-10 h-10 text-blue-600 animate-spin" />
                        <p className="text-slate-500 font-medium animate-pulse">Fetching & Parsing Metadata...</p>
                    </div>
                )}

                {data && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="space-y-8"
                    >
                        <SEOScorecard
                            score={data.analysis.score}
                            suggestions={data.analysis.suggestions}
                        />
                        <SocialPreview data={data} />
                    </motion.div>
                )}

                {/* Error State */}
                {error && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                    >
                        <AlertCircle className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">{error}</span>
                    </motion.div>
                )}

            </main>


        </div>
    );
}
