"use client";

import React from "react";
import { motion } from "framer-motion";
import { AlertCircle, AlertTriangle, Info, CheckCircle, TrendingUp } from "lucide-react";

interface Suggestion {
    severity: 'critical' | 'warning' | 'info';
    message: string;
    fix: string;
}

interface SEOScorecardProps {
    score: number;
    suggestions: Suggestion[];
}

export const SEOScorecard: React.FC<SEOScorecardProps> = ({ score, suggestions }) => {

    const getScoreColor = (s: number) => {
        if (s >= 90) return { text: 'text-emerald-600', bg: 'bg-emerald-500', border: 'border-emerald-200' };
        if (s >= 70) return { text: 'text-amber-500', bg: 'bg-amber-500', border: 'border-amber-200' };
        return { text: 'text-red-500', bg: 'bg-red-500', border: 'border-red-200' };
    };

    const colors = getScoreColor(score);
    const circumference = 2 * Math.PI * 40; // r=40
    const offset = circumference - (score / 100) * circumference;

    return (
        <div className="bg-white rounded-3xl p-8 border border-slate-200 shadow-sm flex flex-col md:flex-row gap-8">

            {/* Score Circle */}
            <div className="flex flex-col items-center justify-center md:w-1/3 border-b md:border-b-0 md:border-r border-slate-100 pb-8 md:pb-0 md:pr-8">
                <div className="relative w-40 h-40 mb-4">
                    <svg className="w-full h-full transform -rotate-90">
                        <circle
                            cx="80"
                            cy="80"
                            r="40"
                            stroke="currentColor"
                            strokeWidth="8"
                            fill="transparent"
                            className="text-slate-100"
                        />
                        <motion.circle
                            initial={{ strokeDashoffset: circumference }}
                            animate={{ strokeDashoffset: offset }}
                            transition={{ duration: 1.5, ease: "easeOut" }}
                            cx="80"
                            cy="80"
                            r="40"
                            stroke="currentColor"
                            strokeWidth="8"
                            fill="transparent"
                            strokeDasharray={circumference}
                            strokeLinecap="round"
                            className={colors.text}
                        />
                    </svg>
                    <div className="absolute inset-0 flex flex-col items-center justify-center">
                        <span className={`text-5xl font-black ${colors.text}`}>{score}</span>
                        <span className="text-xs font-bold text-slate-400 uppercase tracking-widest mt-1">SEO Score</span>
                    </div>
                </div>

                <div className="text-center">
                    <h3 className="font-bold text-slate-900 text-lg mb-1">
                        {score >= 90 ? "Excellent!" : score >= 70 ? "Needs Improvement" : "Critical Issues"}
                    </h3>
                    <p className="text-sm text-slate-500">
                        {score >= 90 ? "Your page is optimized for search engines." : "Address the issues below to improve ranking."}
                    </p>
                </div>
            </div>

            {/* Suggestions List */}
            <div className="flex-1 space-y-4">
                <div className="flex items-center gap-2 mb-2">
                    <TrendingUp className="w-5 h-5 text-blue-600" />
                    <h3 className="font-bold text-slate-900">Optimization Checklist</h3>
                </div>

                {suggestions.length === 0 ? (
                    <div className="bg-emerald-50 rounded-xl p-4 flex items-center gap-3 text-emerald-800 border border-emerald-100">
                        <CheckCircle className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">Perfect! No issues found.</span>
                    </div>
                ) : (
                    <div className="space-y-3">
                        {suggestions.map((issue, idx) => (
                            <motion.div
                                initial={{ opacity: 0, x: 20 }}
                                animate={{ opacity: 1, x: 0 }}
                                transition={{ delay: idx * 0.1 }}
                                key={idx}
                                className="flex gap-4 p-4 rounded-xl bg-slate-50 border border-slate-100"
                            >
                                <div className="mt-0.5 max-md:hidden">
                                    {issue.severity === 'critical' && <AlertCircle className="w-5 h-5 text-red-500" />}
                                    {issue.severity === 'warning' && <AlertTriangle className="w-5 h-5 text-amber-500" />}
                                    {issue.severity === 'info' && <Info className="w-5 h-5 text-blue-500" />}
                                </div>
                                <div className="flex-1">
                                    <div className="flex items-center gap-2 mb-0.5">
                                        {/* Mobile Icon */}
                                        <div className="md:hidden">
                                            {issue.severity === 'critical' && <AlertCircle className="w-4 h-4 text-red-500" />}
                                            {issue.severity === 'warning' && <AlertTriangle className="w-4 h-4 text-amber-500" />}
                                            {issue.severity === 'info' && <Info className="w-4 h-4 text-blue-500" />}
                                        </div>
                                        <h4 className="font-bold text-slate-900 text-sm">{issue.message}</h4>
                                        <span className={`text-[10px] font-bold px-1.5 py-0.5 rounded uppercase tracking-wider ml-auto
                                            ${issue.severity === 'critical' ? 'bg-red-100 text-red-700' :
                                                issue.severity === 'warning' ? 'bg-amber-100 text-amber-700' : 'bg-blue-100 text-blue-700'}
                                        `}>
                                            {issue.severity}
                                        </span>
                                    </div>
                                    <p className="text-sm text-slate-500">{issue.fix}</p>
                                </div>
                            </motion.div>
                        ))}
                    </div>
                )}
            </div>

        </div>
    );
};
