"use client";

import React, { useState } from "react";
import { Copy, AlertTriangle, CheckCircle, Search, Facebook, Twitter } from "lucide-react";
import Image from "next/image";

export interface MetaData {
    title: string;
    description: string;
    image: string;
    url: string;
    ogType: string;
    twitterCard: string;
    favicon: string;
    canonical: string;
    siteName: string;
}

interface SocialPreviewProps {
    data: MetaData;
}

export const SocialPreview: React.FC<SocialPreviewProps> = ({ data }) => {
    const [activeTab, setActiveTab] = useState<'google' | 'facebook' | 'twitter'>('google');

    // Validation Helpers
    const isTitleValid = data.title && data.title.length <= 60;
    const isDescValid = data.description && data.description.length <= 160;

    const getLengthColor = (current: number, max: number) => {
        if (current === 0) return "text-slate-400";
        if (current > max) return "text-red-600 font-bold";
        return "text-emerald-600 font-bold";
    };

    const host = data.url ? new URL(data.url).hostname : 'example.com';
    const cleanUrl = data.url;

    return (
        <div className="space-y-8">

            {/* Control Tabs */}
            <div className="bg-slate-100 p-1 rounded-xl flex items-center justify-between md:justify-start gap-2">
                <button
                    onClick={() => setActiveTab('google')}
                    className={`flex-1 md:flex-none px-6 py-2.5 rounded-lg text-sm font-bold flex items-center justify-center gap-2 transition-all
                        ${activeTab === 'google' ? "bg-white text-blue-600 shadow-sm" : "hover:bg-slate-200 text-slate-500"}
                    `}
                >
                    <Search className="w-4 h-4" /> Google
                </button>
                <button
                    onClick={() => setActiveTab('facebook')}
                    className={`flex-1 md:flex-none px-6 py-2.5 rounded-lg text-sm font-bold flex items-center justify-center gap-2 transition-all
                        ${activeTab === 'facebook' ? "bg-[#1877F2] text-white shadow-sm" : "hover:bg-slate-200 text-slate-500"}
                    `}
                >
                    <Facebook className="w-4 h-4" /> Facebook
                </button>
                <button
                    onClick={() => setActiveTab('twitter')}
                    className={`flex-1 md:flex-none px-6 py-2.5 rounded-lg text-sm font-bold flex items-center justify-center gap-2 transition-all
                        ${activeTab === 'twitter' ? "bg-black text-white shadow-sm" : "hover:bg-slate-200 text-slate-500"}
                    `}
                >
                    <Twitter className="w-4 h-4" /> Twitter
                </button>
            </div>

            {/* Validation Bar */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className={`p-4 rounded-xl border ${isTitleValid ? 'bg-emerald-50 border-emerald-100' : 'bg-red-50 border-red-100'}`}>
                    <div className="flex justify-between items-center mb-1">
                        <span className="text-xs font-bold uppercase tracking-wider text-slate-500">Title Length</span>
                        <span className={`text-xs ${getLengthColor(data.title.length, 60)}`}>{data.title.length} / 60</span>
                    </div>
                    <div className="h-2 bg-white rounded-full overflow-hidden border border-slate-100">
                        <div
                            className={`h-full ${data.title.length > 60 ? 'bg-red-500' : 'bg-emerald-500'}`}
                            style={{ width: `${Math.min(100, (data.title.length / 60) * 100)}%` }}
                        />
                    </div>
                </div>

                <div className={`p-4 rounded-xl border ${isDescValid ? 'bg-emerald-50 border-emerald-100' : 'bg-red-50 border-red-100'}`}>
                    <div className="flex justify-between items-center mb-1">
                        <span className="text-xs font-bold uppercase tracking-wider text-slate-500">Description Length</span>
                        <span className={`text-xs ${getLengthColor(data.description.length, 160)}`}>{data.description.length} / 160</span>
                    </div>
                    <div className="h-2 bg-white rounded-full overflow-hidden border border-slate-100">
                        <div
                            className={`h-full ${data.description.length > 160 ? 'bg-red-500' : 'bg-emerald-500'}`}
                            style={{ width: `${Math.min(100, (data.description.length / 160) * 100)}%` }}
                        />
                    </div>
                </div>
            </div>

            {/* PREVIEW AREA */}
            <div className="bg-slate-50 border border-slate-200 rounded-3xl p-8 flex items-center justify-center min-h-[400px]">

                {/* 1. Google Preview */}
                {activeTab === 'google' && (
                    <div className="w-full max-w-[600px] bg-white p-6 rounded-lg shadow-sm font-sans text-left">
                        <div className="flex items-center gap-3 mb-2">
                            <div className="w-7 h-7 bg-slate-100 rounded-full flex items-center justify-center overflow-hidden border border-slate-200">
                                {data.favicon ? (
                                    <img src={data.favicon} alt="Favicon" className="w-4 h-4" />
                                ) : (
                                    <Search className="w-4 h-4 text-slate-400" />
                                )}
                            </div>
                            <div className="flex flex-col">
                                <span className="text-sm text-[#202124] leading-tight">{data.siteName || host}</span>
                                <span className="text-xs text-[#5f6368] leading-tight">{data.url || 'https://example.com'}</span>
                            </div>
                        </div>
                        <h3 className="text-xl text-[#1a0dab] hover:underline cursor-pointer truncate mb-1 leading-snug">
                            {data.title || "Page Title"}
                        </h3>
                        <p className="text-sm text-[#4d5156] line-clamp-2 leading-relaxed">
                            {data.description || "No description provided. Search engines may display text from the page content instead."}
                        </p>
                    </div>
                )}

                {/* 2. Facebook Preview */}
                {activeTab === 'facebook' && (
                    <div className="w-full max-w-[520px] bg-[#F0F2F5] rounded overflow-hidden shadow-sm border border-slate-200 cursor-pointer">
                        <div className="aspect-[1.91/1] bg-slate-200 relative overflow-hidden">
                            {data.image ? (
                                <img src={data.image} alt="OG Image" className="w-full h-full object-cover" />
                            ) : (
                                <div className="flex items-center justify-center h-full text-slate-400 bg-slate-100">
                                    <span className="text-sm font-bold">No Image Found</span>
                                </div>
                            )}
                        </div>
                        <div className="bg-white p-3 border-t border-slate-100">
                            <p className="text-[12px] uppercase text-[#65676B] truncate mb-0.5">{host}</p>
                            <h3 className="text-[16px] font-bold text-[#050505] leading-tight mb-1 line-clamp-2">
                                {data.title || "Page Title"}
                            </h3>
                            <p className="text-[14px] text-[#65676B] line-clamp-1">
                                {data.description || "Page description..."}
                            </p>
                        </div>
                    </div>
                )}

                {/* 3. Twitter Preview */}
                {activeTab === 'twitter' && (
                    <div className="w-full max-w-[500px] border border-[#CFD9DE] rounded-2xl overflow-hidden cursor-pointer bg-white">
                        <div className="aspect-[1.91/1] bg-slate-100 relative overflow-hidden border-b border-[#CFD9DE]">
                            {data.image ? (
                                <img src={data.image} alt="Twitter Image" className="w-full h-full object-cover" />
                            ) : (
                                <div className="flex items-center justify-center h-full text-slate-400 bg-slate-50">
                                    <span className="text-sm font-bold">No Image</span>
                                </div>
                            )}
                        </div>
                        <div className="p-3">
                            <h3 className="text-[15px] font-bold text-[#0F1419] leading-tight mb-0.5 truncate">
                                {data.title || "Page Title"}
                            </h3>
                            <p className="text-[15px] text-[#536471] mb-1 line-clamp-2 leading-snug">
                                {data.description || "Description preview..."}
                            </p>
                            <div className="flex items-center gap-1 text-[15px] text-[#536471]">
                                <LinkIcon className="w-4 h-4" />
                                <span className="truncate">{host}</span>
                            </div>
                        </div>
                    </div>
                )}

            </div>
        </div>
    );
};

// Helper for Twitter Icon
function LinkIcon({ className }: { className?: string }) {
    return <svg className={className} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71" /><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71" /></svg>;
}
