"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Copy, Check, Server, Shield, Globe, ExternalLink } from "lucide-react";
import Link from "next/link";

export interface MXRecord {
    priority: number;
    exchange: string;
    ip: string | null;
    provider: string;
    isReachable: boolean; // Computed by API if possible, or just static "true" if resolved
}

interface MXResultsProps {
    records: MXRecord[];
    domain: string;
}

export const MXResults: React.FC<MXResultsProps> = ({ records, domain }) => {
    const [copiedAll, setCopiedAll] = useState(false);
    const [copiedIp, setCopiedIp] = useState<string | null>(null);

    const copyAll = () => {
        const text = records.map(r => `${r.priority} ${r.exchange} (${r.ip || 'No IP'})`).join('\n');
        navigator.clipboard.writeText(text);
        setCopiedAll(true);
        setTimeout(() => setCopiedAll(false), 2000);
    };

    const copyIp = (ip: string) => {
        navigator.clipboard.writeText(ip);
        setCopiedIp(ip);
        setTimeout(() => setCopiedIp(null), 2000);
    };

    // Sort by priority
    const sortedRecords = [...records].sort((a, b) => a.priority - b.priority);

    // Identify primary provider (simplistic: take provider of lowest priority)
    const primaryProvider = sortedRecords.length > 0 ? sortedRecords[0].provider : "Unknown";

    const getProviderColor = (provider: string) => {
        if (provider.includes("Google")) return "bg-blue-100 text-blue-700 border-blue-200";
        if (provider.includes("Microsoft") || provider.includes("Outlook")) return "bg-sky-100 text-sky-700 border-sky-200";
        if (provider.includes("Zoho")) return "bg-yellow-100 text-yellow-700 border-yellow-200";
        if (provider.includes("Mimecast")) return "bg-purple-100 text-purple-700 border-purple-200";
        if (provider.includes("Proton")) return "bg-violet-100 text-violet-700 border-violet-200";
        return "bg-slate-100 text-slate-600 border-slate-200";
    };

    return (
        <div className="space-y-6">

            {/* Summary Card */}
            <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-xl shadow-blue-500/5 flex flex-col sm:flex-row items-center justify-between gap-4">
                <div className="flex items-center gap-4">
                    <div className="p-3 bg-blue-50 rounded-xl">
                        <Server className="w-6 h-6 text-blue-600" />
                    </div>
                    <div>
                        <h3 className="text-lg font-bold text-slate-900 leading-none">Mail Servers Found</h3>
                        <p className="text-sm text-slate-500 mt-1 font-medium">
                            {records.length} valid MX records for <span className="text-slate-900">{domain}</span>
                        </p>
                    </div>
                </div>

                <div className="flex items-center gap-3">
                    {primaryProvider !== "Unknown" && (
                        <span className={`px-4 py-1.5 rounded-full text-xs font-bold uppercase tracking-wider border ${getProviderColor(primaryProvider)}`}>
                            {primaryProvider}
                        </span>
                    )}
                    <button
                        onClick={copyAll}
                        className="flex items-center gap-2 px-4 py-2 bg-slate-50 hover:bg-slate-100 text-slate-600 rounded-xl text-sm font-bold transition-colors border border-slate-200"
                    >
                        {copiedAll ? <Check className="w-4 h-4 text-emerald-500" /> : <Copy className="w-4 h-4" />}
                        Copy Report
                    </button>
                </div>
            </div>

            {/* Records List */}
            <div className="space-y-4">
                {sortedRecords.map((record, index) => (
                    <motion.div
                        key={index}
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: index * 0.1 }}
                        className="bg-white border border-slate-200 rounded-2xl p-5 shadow-sm hover:shadow-md hover:border-blue-300 transition-all group flex flex-col md:flex-row md:items-center gap-4"
                    >
                        {/* Priority Badge */}
                        <div className="flex-shrink-0">
                            <div className="w-12 h-12 rounded-xl bg-slate-50 border border-slate-100 flex flex-col items-center justify-center">
                                <span className="text-[10px] uppercase font-bold text-slate-400">Prio</span>
                                <span className="text-lg font-black text-slate-900 leading-none">{record.priority}</span>
                            </div>
                        </div>

                        {/* Hostname & Provider */}
                        <div className="flex-1 min-w-0">
                            <div className="flex items-center gap-2 mb-1">
                                <h4 className="text-base font-bold text-slate-900 truncate font-mono">
                                    {record.exchange}
                                </h4>
                                {record.isReachable && (
                                    <div className="w-2 h-2 rounded-full bg-emerald-500 animate-pulse" title="Active / Reachable" />
                                )}
                            </div>
                            <div className="flex items-center gap-2">
                                <span className={`text-[10px] font-bold px-2 py-0.5 rounded border ${getProviderColor(record.provider)}`}>
                                    {record.provider !== "Unknown" ? record.provider : "Custom / Self-Hosted"}
                                </span>
                            </div>
                        </div>

                        {/* IP & Actions */}
                        <div className="flex items-center gap-4">
                            {record.ip ? (
                                <div className="flex items-center gap-2 bg-slate-50 px-3 py-1.5 rounded-lg border border-slate-200 group-hover:bg-white group-hover:border-blue-200 transition-colors">
                                    <Globe className="w-3.5 h-3.5 text-slate-400" />
                                    <span className="font-mono text-sm font-medium text-slate-600">{record.ip}</span>
                                    <button
                                        onClick={() => copyIp(record.ip!)}
                                        className="ml-2 pl-2 border-l border-slate-200 text-slate-400 hover:text-blue-600 transition-colors"
                                    >
                                        {copiedIp === record.ip ? <Check className="w-3.5 h-3.5" /> : <Copy className="w-3.5 h-3.5" />}
                                    </button>
                                </div>
                            ) : (
                                <span className="text-xs text-amber-500 font-medium italic">IP Not Resolved</span>
                            )}

                            <Link
                                href={record.ip ? `/ip-info?ip=${record.ip}` : '#'}
                                className={`p-2 rounded-lg text-slate-400 transition-colors ${record.ip ? 'hover:bg-blue-50 hover:text-blue-600' : 'cursor-not-allowed opacity-50'}`}
                                title="Analyze IP"
                            >
                                <ExternalLink className="w-4 h-4" />
                            </Link>
                        </div>

                    </motion.div>
                ))}
            </div>

        </div>
    );
};
