"use client";

import React, { useState, useRef, useEffect } from "react";
import { motion } from "framer-motion";
import { Activity, Play, Loader2, Globe, Laptop, Flag } from "lucide-react";

import { PingChart } from "@/components/ping/PingChart";

export interface PingResult {
    seq: number;
    time: number;
    ttl: number;
    status: 'success' | 'timeout' | 'error';
    raw: string;
}

export default function PingClient() {
    const [host, setHost] = useState("");
    const [count, setCount] = useState(10);
    const [running, setRunning] = useState(false);
    const [results, setResults] = useState<PingResult[]>([]);
    const [rawOutput, setRawOutput] = useState<string[]>([]);

    // Stats
    const validResults = results.filter(r => r.status === 'success');
    const minTime = validResults.length ? Math.min(...validResults.map(r => r.time)) : 0;
    const maxTime = validResults.length ? Math.max(...validResults.map(r => r.time)) : 0;
    const avgTime = validResults.length ? validResults.reduce((a, b) => a + b.time, 0) / validResults.length : 0;
    const loss = results.length > 0 ? ((results.length - validResults.length) / results.length) * 100 : 0;

    const terminalEndRef = useRef<HTMLDivElement>(null);

    // Auto-scroll terminal
    useEffect(() => {
        terminalEndRef.current?.scrollIntoView({ behavior: "smooth" });
    }, [rawOutput]);

    const handlePing = async () => {
        if (!host) return;
        setRunning(true);
        setResults([]);
        setRawOutput([]);
        let seq = 0;

        try {
            const response = await fetch('/api/ping', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ host, count }),
            });

            if (!response.body) throw new Error("No response body");

            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            let buffer = '';

            while (true) {
                const { done, value } = await reader.read();
                if (done) break;

                const chunk = decoder.decode(value, { stream: true });
                buffer += chunk;

                // Process lines
                const lines = buffer.split(/\r?\n/);
                // Keep the last partial line in buffer
                buffer = lines.pop() || '';

                for (const line of lines) {
                    if (!line.trim()) continue;
                    setRawOutput(prev => [...prev, line]);

                    // Parse Logic
                    // Linux: 64 bytes from 8.8.8.8: icmp_seq=1 ttl=118 time=15.2 ms
                    // Windows: Reply from 8.8.8.8: bytes=32 time=15ms TTL=118

                    const timeMatch = line.match(/(?:time=|time<)([\d\.]+)\s*ms?/i);
                    const ttlMatch = line.match(/TTL=(\d+)/i);

                    if (timeMatch) {
                        seq++;
                        const time = parseFloat(timeMatch[1]);
                        const ttl = ttlMatch ? parseInt(ttlMatch[1]) : 0;
                        setResults(prev => [...prev, {
                            seq,
                            time,
                            ttl,
                            status: 'success',
                            raw: line
                        }]);
                    } else if (line.includes("Request timed out") || line.includes("100% packet loss")) {
                        seq++;
                        setResults(prev => [...prev, {
                            seq,
                            time: 0,
                            ttl: 0,
                            status: 'timeout',
                            raw: line
                        }]);
                    }
                }
            }

        } catch (error) {
            console.error("Ping Error:", error);
            setRawOutput(prev => [...prev, `\nError: ${error}`]);
        } finally {
            setRunning(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Activity className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                High-Precision <span className="text-blue-600">Ping Test</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Measure network latency, packet loss, and jitter with real-time ICMP tracking.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Controls */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <div className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Host (e.g., 1.1.1.1 or google.com)"
                                value={host}
                                onChange={(e) => setHost(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <div className="w-full md:w-48 relative">
                            <select
                                value={count}
                                onChange={(e) => setCount(Number(e.target.value))}
                                className="w-full h-14 pl-4 pr-10 bg-slate-50 border border-slate-200 rounded-2xl font-bold text-slate-700 outline-none focus:border-blue-500 appearance-none cursor-pointer"
                            >
                                <option value="4">4 Packets</option>
                                <option value="10">10 Packets</option>
                                <option value="20">20 Packets</option>
                                <option value="50">50 Packets</option>
                            </select>
                            <Flag className="absolute right-4 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-400 pointer-events-none" />
                        </div>

                        <button
                            onClick={handlePing}
                            disabled={running || !host}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${running ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {running ? <Loader2 className="w-5 h-5 animate-spin" /> : <Play className="w-5 h-5 fill-current" />}
                            {running ? "Pinging..." : "Start Ping"}
                        </button>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 h-[600px]">

                    {/* Left: Stats & Terminal */}
                    <div className="col-span-1 flex flex-col gap-6 h-full">

                        {/* Big Stats */}
                        <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-xl shadow-blue-500/5 grid grid-cols-2 gap-4">
                            <StatBox label="Min Latency" value={minTime ? `${minTime.toFixed(1)}ms` : "--"} color="text-emerald-600" />
                            <StatBox label="Max Latency" value={maxTime ? `${maxTime.toFixed(1)}ms` : "--"} color="text-rose-600" />
                            <StatBox label="Avg Latency" value={avgTime ? `${avgTime.toFixed(1)}ms` : "--"} color="text-blue-600" />
                            <StatBox label="Packet Loss" value={`${loss.toFixed(0)}%`} color={loss > 0 ? "text-amber-600" : "text-emerald-600"} />
                        </div>

                        {/* Terminal */}
                        <div className="flex-1 bg-slate-900 rounded-3xl p-6 shadow-inner overflow-hidden flex flex-col font-mono text-sm relative">
                            <div className="flex items-center gap-2 mb-4 text-slate-400 border-b border-slate-800 pb-2">
                                <Laptop className="w-4 h-4" />
                                <span className="text-xs font-bold uppercase tracking-widest">Console Output</span>
                            </div>
                            <div className="overflow-y-auto flex-1 scrollbar-thin scrollbar-thumb-slate-700 scrollbar-track-transparent pr-2">
                                {rawOutput.length === 0 && !running && (
                                    <div className="text-slate-600 italic">Ready to ping...</div>
                                )}
                                {rawOutput.map((line, i) => (
                                    <div key={i} className="mb-0.5 whitespace-pre-wrap break-all">
                                        {line.includes("Reply from") || line.includes("bytes from") ? (
                                            <span className="text-emerald-400">{line}</span>
                                        ) : line.includes("Request timed out") ? (
                                            <span className="text-amber-500 font-bold">{line}</span>
                                        ) : line.includes("Error") ? (
                                            <span className="text-red-500 font-bold">{line}</span>
                                        ) : (
                                            <span className="text-slate-300">{line}</span>
                                        )}
                                    </div>
                                ))}
                                <div ref={terminalEndRef} />
                            </div>
                        </div>

                    </div>

                    {/* Right: Chart */}
                    <div className="col-span-1 lg:col-span-2 h-full">
                        <PingChart data={results} />
                    </div>

                </div>

            </main>


        </div>
    );
}

const StatBox = ({ label, value, color }: any) => (
    <div className="flex flex-col">
        <span className="text-[10px] font-bold text-slate-400 uppercase tracking-wider mb-0.5">{label}</span>
        <span className={`text-xl font-black ${color}`}>{value}</span>
    </div>
);
