"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Loader2, Play, Settings2, AlertCircle } from "lucide-react";

import { PortTable, PortResult } from "@/components/port-scanner/PortTable";

// Presets
const COMMON_PORTS = [21, 22, 23, 25, 53, 80, 110, 135, 139, 143, 443, 445, 993, 995, 1433, 3306, 3389, 5432, 5900, 6379, 8080, 8443, 27017];
const WEB_PORTS = [80, 443, 8000, 8008, 8080, 8443, 8888, 9000, 9090];
// Top 50 most common nmap ports (subset)
const TOP_50_PORTS = [21, 22, 23, 25, 53, 80, 110, 135, 139, 143, 443, 445, 993, 995, 1723, 3306, 3389, 5900, 8080, ...WEB_PORTS].filter((v, i, a) => a.indexOf(v) === i).sort((a, b) => a - b);

export default function PortScannerClient() {
    const [target, setTarget] = useState("");
    const [preset, setPreset] = useState("common");
    const [customRange, setCustomRange] = useState("1-100");
    const [isScanning, setIsScanning] = useState(false);
    const [results, setResults] = useState<PortResult[]>([]);
    const [progress, setProgress] = useState(0);
    const [error, setError] = useState("");

    const getPorts = () => {
        if (preset === "common") return COMMON_PORTS;
        if (preset === "web") return WEB_PORTS;
        if (preset === "top50") return TOP_50_PORTS;
        if (preset === "custom") {
            // Parse range "1-100" or simple "80,443"
            if (customRange.includes("-")) {
                const [start, end] = customRange.split("-").map(Number);
                if (!start || !end || end < start) return [];
                const p = [];
                // Limit range size to avoid abuse/timeout
                const safeEnd = Math.min(end, start + 100);
                for (let i = start; i <= safeEnd; i++) p.push(i);
                return p;
            }
            return customRange.split(",").map(p => parseInt(p.trim())).filter(p => !isNaN(p));
        }
        return COMMON_PORTS;
    };

    const handleScan = async () => {
        if (!target) return;
        setIsScanning(true);
        setResults([]);
        setProgress(0);
        setError("");

        const portsToScan = getPorts();
        const batchSize = 10;
        const total = portsToScan.length;

        try {
            // Chunking
            for (let i = 0; i < total; i += batchSize) {
                const batch = portsToScan.slice(i, i + batchSize);

                const res = await fetch('/api/scan', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ target, ports: batch })
                });

                if (!res.ok) {
                    const err = await res.json();
                    throw new Error(err.error || "Scan failed");
                }

                const data = await res.json();
                setResults(prev => [...prev, ...data.results]);

                // Update progress
                setProgress(Math.min(100, Math.round(((i + batch.length) / total) * 100)));
            }
        } catch (err: any) {
            setError(err.message || "An error occurred during scanning");
        } finally {
            setIsScanning(false);
            setProgress(100);
        }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">
            {/* Header */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-16 relative overflow-hidden">
                <div className="absolute inset-0 opacity-5 pointer-events-none" style={{
                    backgroundImage: `radial-gradient(circle at 2px 2px, #94a3b8 1px, transparent 0)`,
                    backgroundSize: '24px 24px'
                }} />

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center relative z-10">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.5 }}
                    >
                        <div className="inline-flex items-center justify-center p-3 bg-blue-50 rounded-2xl mb-6 text-[#2563EB]">
                            <Settings2 className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-4 font-jakarta">
                            Professional Port Scanner
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto mb-10">
                            Scan common ports to audit network security and verify firewall rules.
                        </p>

                        {/* Control Bar */}
                        <div className="max-w-4xl mx-auto bg-white p-3 rounded-2xl shadow-[0_20px_40px_-15px_rgba(0,0,0,0.05)] border border-[#E2E8F0] flex flex-col md:flex-row gap-3 items-center">

                            {/* Input */}
                            <div className="w-full md:flex-1 relative">
                                <input
                                    type="text"
                                    placeholder="IP Address or Domain"
                                    value={target}
                                    onChange={(e) => setTarget(e.target.value)}
                                    className="w-full h-12 pl-4 pr-4 bg-slate-50 rounded-xl border-transparent focus:bg-white focus:ring-2 focus:ring-blue-100 outline-none transition-all font-mono text-sm"
                                    suppressHydrationWarning
                                />
                            </div>

                            {/* Presets */}
                            <div className="w-full md:w-auto relative">
                                <select
                                    value={preset}
                                    onChange={(e) => setPreset(e.target.value)}
                                    className="w-full h-12 pl-4 pr-10 appearance-none bg-slate-50 rounded-xl border-transparent focus:bg-white focus:ring-2 focus:ring-blue-100 outline-none font-bold text-slate-700 text-sm cursor-pointer"
                                >
                                    <option value="common">Common Ports</option>
                                    <option value="web">Web Ports (80, 443...)</option>
                                    <option value="top50">Top 50 Ports</option>
                                    <option value="custom">Custom Range</option>
                                </select>
                            </div>

                            {/* Custom Range Input (Conditional) */}
                            {preset === "custom" && (
                                <div className="w-full md:w-32">
                                    <input
                                        type="text"
                                        placeholder="1-100"
                                        value={customRange}
                                        onChange={(e) => setCustomRange(e.target.value)}
                                        className="w-full h-12 px-4 bg-slate-50 rounded-xl font-mono text-sm outline-none focus:ring-2 focus:ring-blue-100"
                                    />
                                </div>
                            )}

                            {/* Action Button */}
                            <button
                                onClick={handleScan}
                                disabled={isScanning || !target}
                                className="w-full md:w-auto h-12 px-8 bg-[#2563EB] hover:bg-[#1E3A8A] text-white font-bold rounded-xl transition-all shadow-lg active:scale-95 disabled:opacity-70 disabled:cursor-not-allowed flex items-center justify-center gap-2 whitespace-nowrap"
                            >
                                {isScanning ? <Loader2 className="w-5 h-5 animate-spin" /> : <Play className="w-5 h-5 fill-current" />}
                                Start Scan
                            </button>
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Progress Bar */}
            {isScanning && (
                <div className="h-1.5 w-full bg-slate-100 overflow-hidden">
                    <motion.div
                        className="h-full bg-[#2563EB]"
                        initial={{ width: 0 }}
                        animate={{ width: `${progress}%` }}
                        transition={{ type: "tween", ease: "easeOut" }}
                    />
                </div>
            )}

            {/* Results Area */}
            <main className="flex-1 max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-12 w-full relative z-10">
                {error && (
                    <div className="bg-red-50 border border-red-100 rounded-xl p-4 flex items-center gap-3 text-red-700 mb-6">
                        <AlertCircle className="w-5 h-5 shrink-0" />
                        <span className="font-medium">{error}</span>
                    </div>
                )}

                <PortTable results={results} isScanning={isScanning} />
            </main>


        </div>
    );
}
