"use client";

import React from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Download, ShieldCheck, ShieldAlert, Monitor } from "lucide-react";

export interface PortResult {
    port: number;
    service: string;
    status: 'open' | 'closed' | 'filtered';
    latency: string;
}

interface PortTableProps {
    results: PortResult[];
    isScanning: boolean;
}

export const PortTable: React.FC<PortTableProps> = ({ results, isScanning }) => {

    const downloadCsv = () => {
        const headers = ["Port", "Service", "Status", "Latency"];
        const rows = results.map(r => [r.port, r.service, r.status, r.latency].join(","));
        const csvContent = [headers.join(","), ...rows].join("\n");

        const blob = new Blob([csvContent], { type: "text/csv" });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url;
        a.download = `port-scan-results-${new Date().toISOString()}.csv`;
        a.click();
    };

    if (results.length === 0 && !isScanning) {
        return (
            <div className="flex flex-col items-center justify-center py-20 text-center opacity-50">
                <Monitor className="w-16 h-16 text-slate-300 mb-4" />
                <h3 className="text-xl font-bold text-slate-400">Ready to Audit</h3>
                <p className="text-slate-400 mt-2">Enter a target and start scanning to see results.</p>
            </div>
        );
    }

    return (
        <div className="w-full bg-white border border-slate-200 rounded-2xl shadow-sm overflow-hidden flex flex-col">
            {/* Toolbar */}
            <div className="p-4 border-b border-slate-200 flex justify-between items-center bg-slate-50">
                <div className="text-sm font-bold text-slate-700">
                    Results ({results.length})
                </div>
                {results.length > 0 && (
                    <button
                        onClick={downloadCsv}
                        className="flex items-center gap-2 text-xs font-bold text-blue-600 hover:text-blue-700 bg-blue-50 px-3 py-2 rounded-lg transition-colors"
                    >
                        <Download className="w-4 h-4" /> Export CSV
                    </button>
                )}
            </div>

            {/* Table */}
            <div className="overflow-x-auto">
                <table className="w-full text-left border-collapse">
                    <thead>
                        <tr>
                            <th className="p-4 text-xs font-bold text-slate-900 border-b border-slate-200 uppercase tracking-wider w-24">Port</th>
                            <th className="p-4 text-xs font-bold text-slate-900 border-b border-slate-200 uppercase tracking-wider">Service</th>
                            <th className="p-4 text-xs font-bold text-slate-900 border-b border-slate-200 uppercase tracking-wider">Status</th>
                            <th className="p-4 text-xs font-bold text-slate-900 border-b border-slate-200 uppercase tracking-wider text-right">Latency</th>
                        </tr>
                    </thead>
                    <tbody className="text-sm font-mono text-slate-600">
                        <AnimatePresence initial={false}>
                            {results.map((result, index) => (
                                <motion.tr
                                    key={result.port}
                                    initial={{ opacity: 0, x: -20 }}
                                    animate={{ opacity: 1, x: 0 }}
                                    transition={{ duration: 0.2, delay: index * 0.05 }} // Stagger slighty
                                    className={`${index % 2 === 0 ? "bg-white" : "bg-slate-50/50"} hover:bg-blue-50/20 transition-colors`}
                                >
                                    <td className="p-4 font-bold text-slate-800">{result.port}</td>
                                    <td className="p-4">{result.service}</td>
                                    <td className="p-4">
                                        {result.status === "open" ? (
                                            <span className="inline-flex items-center gap-1.5 bg-green-100 text-green-700 px-2.5 py-1 rounded-full text-xs font-bold uppercase">
                                                <div className="w-2 h-2 rounded-full bg-green-500 animate-pulse" />
                                                Open
                                            </span>
                                        ) : (
                                            <span className="inline-flex items-center gap-1.5 bg-slate-100 text-slate-400 px-2.5 py-1 rounded-full text-xs font-bold uppercase">
                                                <div className="w-2 h-2 rounded-full bg-slate-300" />
                                                {result.status}
                                            </span>
                                        )}
                                    </td>
                                    <td className="p-4 text-right text-slate-400">{result.latency}</td>
                                </motion.tr>
                            ))}
                        </AnimatePresence>
                    </tbody>
                </table>
            </div>
        </div>
    );
};
