"use client";

import React from "react";
import { motion } from "framer-motion";
import { ShieldCheck, Calendar, Lock, Server, Link as LinkIcon, AlertTriangle } from "lucide-react";

export interface SSLData {
    subject: string;
    issuer: string;
    validFrom: string;
    validTo: string;
    daysRemaining: number;
    serialNumber: string;
    fingerprint: string; // SHA256 of cert
    cipher: string;
    protocol: string;
    chain: string[]; // List of Common Names in chain
    authorized: boolean;
    error?: string;
}

interface SSLDetailsProps {
    data: SSLData;
}

export const SSLDetails: React.FC<SSLDetailsProps> = ({ data }) => {

    // Calculate percentage for circular progress (assuming 90 days or 365 days max for visual mostly)
    // Most certs are 90 days (LetsEncrypt) or 398 days max.
    const maxDays = 365;
    const percentage = Math.max(0, Math.min(100, (data.daysRemaining / maxDays) * 100));
    const circleCircumference = 2 * Math.PI * 40; // r=40
    const strokeDashoffset = circleCircumference - (percentage / 100) * circleCircumference;

    const isExpiringSoon = data.daysRemaining < 14;
    const isValid = data.authorized && data.daysRemaining > 0;

    return (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 w-full">

            {/* 1. Validity Card */}
            <motion.div
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.1 }}
                className="bg-[#F8FAFC] border border-slate-200 rounded-2xl p-6 flex flex-col items-center justify-center relative overflow-hidden"
            >
                <div className="absolute top-4 left-4 text-xs font-bold text-slate-500 uppercase tracking-wider">Validity</div>

                <div className="relative w-32 h-32 flex items-center justify-center mb-4">
                    <svg className="w-full h-full transform -rotate-90">
                        <circle cx="64" cy="64" r="40" stroke="#E2E8F0" strokeWidth="8" fill="transparent" />
                        <circle
                            cx="64" cy="64" r="40"
                            stroke={isValid ? (isExpiringSoon ? "#EAB308" : "#22C55E") : "#EF4444"}
                            strokeWidth="8"
                            fill="transparent"
                            strokeDasharray={circleCircumference}
                            strokeDashoffset={strokeDashoffset}
                            strokeLinecap="round"
                        />
                    </svg>
                    <div className="absolute flex flex-col items-center">
                        <span className={`text-2xl font-extrabold ${isValid ? "text-slate-700" : "text-red-500"}`}>
                            {data.daysRemaining}
                        </span>
                        <span className="text-xs text-slate-400 font-medium">days left</span>
                    </div>
                </div>

                <div className="w-full flex justify-between text-sm mt-2 px-4">
                    <div className="flex flex-col">
                        <span className="text-slate-400 text-xs">Issued On</span>
                        <span className="font-semibold text-slate-700">{new Date(data.validFrom).toLocaleDateString()}</span>
                    </div>
                    <div className="flex flex-col text-right">
                        <span className="text-slate-400 text-xs">Expires On</span>
                        <span className={`font-semibold ${isExpiringSoon || !isValid ? "text-red-600" : "text-slate-700"}`}>
                            {new Date(data.validTo).toLocaleDateString()}
                        </span>
                    </div>
                </div>
            </motion.div>

            {/* 2. Issuer Info */}
            <motion.div
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.2 }}
                className="bg-[#F8FAFC] border border-slate-200 rounded-2xl p-6 relative"
            >
                <div className="text-xs font-bold text-slate-500 uppercase tracking-wider mb-6 flex items-center gap-2">
                    <ShieldCheck className="w-4 h-4" /> Issuer Information
                </div>

                <div className="space-y-4">
                    <div>
                        <div className="text-xs text-slate-400 mb-1">Common Name (CN)</div>
                        <div className="font-bold text-slate-800 text-lg break-words">{data.issuer}</div>
                    </div>
                    <div>
                        <div className="text-xs text-slate-400 mb-1">Subject (Site)</div>
                        <div className="font-medium text-slate-700 break-words">{data.subject}</div>
                    </div>
                    <div>
                        <div className="text-xs text-slate-400 mb-1">Serial Number</div>
                        <div className="font-mono text-xs text-slate-500 break-all">{data.serialNumber}</div>
                    </div>
                </div>
            </motion.div>

            {/* 3. Technical Specs */}
            <motion.div
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.3 }}
                className="bg-[#F8FAFC] border border-slate-200 rounded-2xl p-6 relative"
            >
                <div className="text-xs font-bold text-slate-500 uppercase tracking-wider mb-6 flex items-center gap-2">
                    <Lock className="w-4 h-4" /> Security Specs
                </div>

                <div className="space-y-5">
                    <div className="flex items-center justify-between border-b border-slate-100 pb-2">
                        <span className="text-sm text-slate-500">Protocol</span>
                        <span className="font-bold text-blue-600 bg-blue-50 px-2 py-1 rounded text-xs">{data.protocol}</span>
                    </div>
                    <div className="flex items-center justify-between border-b border-slate-100 pb-2">
                        <span className="text-sm text-slate-500">Authorized</span>
                        <span className={`font-bold px-2 py-1 rounded text-xs ${data.authorized ? "text-green-600 bg-green-50" : "text-red-600 bg-red-50"}`}>
                            {data.authorized ? "Trusted" : "Untrusted"}
                        </span>
                    </div>
                    <div>
                        <span className="text-sm text-slate-500 block mb-1">Cipher Suite</span>
                        <span className="font-mono text-xs text-slate-600 bg-slate-100 p-1.5 rounded block break-all">
                            {data.cipher}
                        </span>
                    </div>
                </div>
            </motion.div>

            {/* 4. Certificate Chain */}
            <motion.div
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.4 }}
                className="bg-[#F8FAFC] border border-slate-200 rounded-2xl p-6 relative flex flex-col"
            >
                <div className="text-xs font-bold text-slate-500 uppercase tracking-wider mb-6 flex items-center gap-2">
                    <LinkIcon className="w-4 h-4" /> Chain of Trust
                </div>

                <div className="flex-1 flex flex-col justify-center space-y-3">
                    {data.chain.map((node, index) => (
                        <div key={index} className="flex flex-col items-center">
                            <div className={`w-full p-3 rounded-lg border text-sm font-semibold flex items-center gap-2
                                ${index === 0 ? "border-green-200 bg-green-50 text-green-700" : // Root
                                    index === data.chain.length - 1 ? "border-blue-200 bg-blue-50 text-blue-700" : // Leaf
                                        "border-slate-200 bg-white text-slate-600"} // Intermediate
                            `}>
                                <Server className="w-4 h-4 opacity-50" />
                                <span className="truncate">{node}</span>
                                {index === 0 && <span className="ml-auto text-[10px] uppercase font-bold tracking-wider opacity-60">Root</span>}
                                {index === data.chain.length - 1 && <span className="ml-auto text-[10px] uppercase font-bold tracking-wider opacity-60">Leaf</span>}
                            </div>

                            {/* Connector Line */}
                            {index < data.chain.length - 1 && (
                                <div className="h-4 w-0.5 bg-slate-300 my-0.5" />
                            )}
                        </div>
                    ))}

                    {data.chain.length === 0 && (
                        <div className="text-center text-slate-400 text-sm">No chain information available</div>
                    )}
                </div>
            </motion.div>

        </div>
    );
};
