"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    ChevronDown, ChevronUp, Copy, Check, Server,
    AlertTriangle, Activity, Globe
} from "lucide-react";

export interface HopData {
    hop: number;
    host: string;
    ip: string;
    rtt: number[]; // Array of RTTs for this hop
    loss: number;
    avg: number;
    best: number;
    worst: number;
    stdev: number;
    status: 'pending' | 'active' | 'done' | 'timeout';
}

interface TraceResultsProps {
    hops: HopData[];
    rawOutput: string[];
    isTracing: boolean;
}

export const TraceResults: React.FC<TraceResultsProps> = ({ hops, rawOutput, isTracing }) => {
    const [showRaw, setShowRaw] = useState(false);
    const [copiedIp, setCopiedIp] = useState<string | null>(null);

    const copyToClipboard = (text: string) => {
        navigator.clipboard.writeText(text);
        setCopiedIp(text);
        setTimeout(() => setCopiedIp(null), 2000);
    };

    return (
        <div className="space-y-8">

            {/* 1. Visual Path (Antigravity Style) */}
            <div className="bg-white border border-slate-200 rounded-3xl p-8 shadow-xl shadow-blue-500/5 overflow-hidden relative">
                <div className="flex items-center gap-3 mb-6">
                    <div className="p-2 bg-blue-50 rounded-lg">
                        <Activity className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                        <h3 className="text-lg font-bold text-slate-900 leading-none">Hop Visualization</h3>
                        <p className="text-xs text-slate-500 font-medium mt-1">Network Path Topology</p>
                    </div>
                </div>

                <div className="relative w-full overflow-x-auto pb-4 scrollbar-thin scrollbar-thumb-slate-200 scrollbar-track-transparent">
                    <div className="flex items-center min-w-max px-4 pt-4 pb-8">
                        {/* Start Node */}
                        <div className="flex flex-col items-center relative z-10">
                            <div className="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center border-4 border-white shadow-lg z-10">
                                <Globe className="w-5 h-5 text-blue-600" />
                            </div>
                            <span className="text-xs font-bold text-slate-500 mt-2">You</span>
                        </div>

                        {hops.map((hop, index) => (
                            <React.Fragment key={hop.hop}>
                                {/* Connection Line */}
                                <motion.div
                                    initial={{ width: 0, opacity: 0 }}
                                    animate={{ width: 60, opacity: 1 }}
                                    className="h-1 bg-slate-100 relative -mx-1"
                                >
                                    <motion.div
                                        initial={{ width: 0 }}
                                        animate={{ width: "100%" }}
                                        transition={{ duration: 0.5, delay: 0.2 }}
                                        className={`h-full ${hop.status === 'timeout' ? 'bg-amber-200' : 'bg-blue-200'}`}
                                    />
                                </motion.div>

                                {/* Hop Node */}
                                <motion.div
                                    initial={{ scale: 0, opacity: 0 }}
                                    animate={{ scale: 1, opacity: 1 }}
                                    transition={{ type: "spring", stiffness: 200, damping: 15 }}
                                    className="flex flex-col items-center relative z-10 group"
                                >
                                    <div className={`
                                        w-10 h-10 rounded-full flex items-center justify-center border-4 border-white shadow-lg transition-all
                                        ${hop.status === 'active' ? 'bg-blue-600 text-white scale-110 shadow-blue-500/30 ring-4 ring-blue-100' :
                                            hop.status === 'timeout' ? 'bg-amber-100 text-amber-600' :
                                                'bg-slate-50 text-slate-500 hover:bg-white hover:text-blue-600'}
                                    `}>
                                        <span className="text-xs font-bold font-mono">{hop.hop}</span>
                                    </div>

                                    {/* Tooltipish Info */}
                                    <div className="absolute top-12 left-1/2 -translate-x-1/2 bg-slate-900 text-white text-[10px] py-1 px-2 rounded opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-20 pointer-events-none">
                                        {hop.ip || "Request Timed Out"}
                                    </div>

                                    <div className="mt-2 text-[10px] font-bold text-slate-400 font-mono">
                                        {hop.avg > 0 ? `${hop.avg.toFixed(0)}ms` : "*"}
                                    </div>
                                </motion.div>
                            </React.Fragment>
                        ))}

                        {/* Destination (if presumably finished/tracing) */}
                        {isTracing && (
                            <>
                                <div className="w-16 h-1 bg-slate-100 -mx-1 animate-pulse" />
                                <div className="w-3 h-3 rounded-full bg-slate-200 animate-ping" />
                            </>
                        )}
                    </div>
                </div>
            </div>

            {/* 2. Detailed Table */}
            <div className="bg-white border border-slate-200 rounded-3xl shadow-xl shadow-blue-500/5 overflow-hidden">
                <div className="p-6 border-b border-slate-50 flex items-center justify-between">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-slate-50 rounded-lg">
                            <Server className="w-5 h-5 text-slate-600" />
                        </div>
                        <h3 className="text-lg font-bold text-slate-900 leading-none">Route Metrics</h3>
                    </div>
                    <div className="text-xs font-bold text-slate-400 uppercase tracking-widest">{hops.length} Hops Discovered</div>
                </div>

                <div className="overflow-x-auto">
                    <table className="w-full text-left border-collapse">
                        <thead>
                            <tr className="bg-slate-50/50 text-slate-500 text-xs uppercase tracking-wider font-bold">
                                <th className="p-4 border-b border-slate-100">#</th>
                                <th className="p-4 border-b border-slate-100 w-1/3">Host / IP</th>
                                <th className="p-4 border-b border-slate-100 text-right">Loss%</th>
                                <th className="p-4 border-b border-slate-100 text-right">Last</th>
                                <th className="p-4 border-b border-slate-100 text-right">Avg</th>
                                <th className="p-4 border-b border-slate-100 text-right">Best</th>
                                <th className="p-4 border-b border-slate-100 text-right">Worst</th>
                                <th className="p-4 border-b border-slate-100 text-right">StDev</th>
                            </tr>
                        </thead>
                        <tbody className="text-sm font-medium text-slate-700 divide-y divide-slate-50">
                            {hops.map((hop) => (
                                <tr key={hop.hop} className="hover:bg-blue-50/30 transition-colors group">
                                    <td className="p-4 font-mono text-slate-400">{hop.hop}</td>
                                    <td className="p-4">
                                        <div className="flex items-center gap-2">
                                            {hop.status === 'timeout' ? (
                                                <span className="text-amber-500 italic flex items-center gap-1">
                                                    <AlertTriangle className="w-3 h-3" /> Request Timed Out
                                                </span>
                                            ) : (
                                                <>
                                                    <span className="font-mono text-slate-900">{hop.ip}</span>
                                                    {hop.host !== hop.ip && hop.host && (
                                                        <span className="text-xs text-slate-400 hidden sm:inline-block">({hop.host})</span>
                                                    )}
                                                    <button
                                                        onClick={() => copyToClipboard(hop.ip)}
                                                        className="opacity-0 group-hover:opacity-100 transition-opacity p-1 hover:bg-slate-100 rounded"
                                                    >
                                                        {copiedIp === hop.ip ? <Check className="w-3 h-3 text-emerald-500" /> : <Copy className="w-3 h-3 text-slate-400" />}
                                                    </button>
                                                </>
                                            )}
                                        </div>
                                    </td>
                                    <td className="p-4 text-right">
                                        <span className={`${hop.loss > 0 ? 'text-amber-600 bg-amber-50 px-2 py-0.5 rounded' : 'text-slate-400'}`}>
                                            {hop.loss}%
                                        </span>
                                    </td>
                                    <td className="p-4 text-right font-mono">
                                        {hop.rtt.length > 0 ? `${hop.rtt[hop.rtt.length - 1].toFixed(1)}ms` : '-'}
                                    </td>
                                    <td className="p-4 text-right font-mono font-bold text-slate-900">
                                        {hop.avg > 0 ? `${hop.avg.toFixed(1)}ms` : '-'}
                                    </td>
                                    <td className="p-4 text-right font-mono text-emerald-600">
                                        {hop.best < 9999 ? `${hop.best.toFixed(1)}ms` : '-'}
                                    </td>
                                    <td className="p-4 text-right font-mono text-rose-600">
                                        {hop.worst > 0 ? `${hop.worst.toFixed(1)}ms` : '-'}
                                    </td>
                                    <td className="p-4 text-right font-mono text-slate-400">
                                        {hop.stdev > 0 ? `±${hop.stdev.toFixed(1)}` : '-'}
                                    </td>
                                </tr>
                            ))}
                            {hops.length === 0 && (
                                <tr>
                                    <td colSpan={8} className="p-8 text-center text-slate-400 italic">
                                        Start a trace to visualize the route...
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* 3. Raw Output Toggle */}
            <div className="border border-slate-200 rounded-2xl overflow-hidden bg-slate-900">
                <button
                    onClick={() => setShowRaw(!showRaw)}
                    className="w-full flex items-center justify-between p-4 text-slate-400 hover:text-white transition-colors text-sm font-bold uppercase tracking-wider bg-slate-900"
                >
                    <span>Raw Terminal Output</span>
                    {showRaw ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
                </button>

                <AnimatePresence>
                    {showRaw && (
                        <motion.div
                            initial={{ height: 0 }}
                            animate={{ height: "auto" }}
                            exit={{ height: 0 }}
                            className="overflow-hidden"
                        >
                            <div className="p-4 pt-0 font-mono text-xs text-slate-300 max-h-64 overflow-y-auto scrollbar-thin scrollbar-thumb-slate-700">
                                {rawOutput.map((line, i) => (
                                    <div key={i} className="mb-0.5">{line}</div>
                                ))}
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </div>

        </div>
    );
};
